<?php
/**
 * MAFIA SINGLE SENDS ROTATION
 *
 * Professional Email Marketing System with Installation Wizard
 * 
 * Features:
 * - Database Installation Wizard
 * - Admin Authentication System
 * - Multi-profile rotation sending
 * - Real-time campaign progress tracking
 * - Contact list management
 * - Email tracking (opens, clicks, bounces)
 * 
 * Enhanced with:
 * - First-time setup wizard with professional UI
 * - Secure admin login
 * - Automatic database table creation
 * - Fixed campaign status bug (campaigns no longer stuck at "sending")
 */

///////////////////////
//  LOAD CONFIGURATION
///////////////////////
// Check if config file exists
$configFile = __DIR__ . '/config.php';
$configExists = file_exists($configFile);

// If config doesn't exist and not installing, show installation wizard
if (!$configExists && (!isset($_GET['action']) || $_GET['action'] !== 'install')) {
    // Show installation wizard
    include_once 'install_wizard.php';
    exit;
}

// Load configuration if it exists
if ($configExists) {
    require_once $configFile;
    // Set globals for CLI workers
    $DB_HOST = defined('DB_HOST') ? DB_HOST : 'localhost';
    $DB_NAME = defined('DB_NAME') ? DB_NAME : '';
    $DB_USER = defined('DB_USER') ? DB_USER : '';
    $DB_PASS = defined('DB_PASS') ? DB_PASS : '';
} else {
    // Defaults for installation
    $DB_HOST = 'localhost';
    $DB_NAME = '';
    $DB_USER = '';
    $DB_PASS = '';
}

///////////////////////
//  CONSTANTS
///////////////////////
// Brand configuration
define('BRAND_NAME', 'MAFIA MAILER');

// Worker configuration defaults (kept for backward compatibility)
define('DEFAULT_WORKERS', 4);
define('DEFAULT_MESSAGES_PER_WORKER', 100);
define('MIN_WORKERS', 1);
define('MAX_WORKERS', 20);
define('MIN_MESSAGES_PER_WORKER', 1);

// Connection Numbers configuration (MaxBulk Mailer style)
define('DEFAULT_CONNECTIONS', 5);
define('MIN_CONNECTIONS', 1);
define('MAX_CONNECTIONS', 40);

// Batch size configuration (emails per single SMTP connection)
define('DEFAULT_BATCH_SIZE', 50);
define('MIN_BATCH_SIZE', 1);
define('MAX_BATCH_SIZE', 500);

// Progress update frequency (update progress every N emails)
define('PROGRESS_UPDATE_FREQUENCY', 10);

///////////////////////
//  DATABASE CONNECTION
///////////////////////
$pdo = null;
if ($DB_NAME !== '' && $DB_USER !== '') {
    try {
        $pdo = new PDO(
            "mysql:host={$DB_HOST};dbname={$DB_NAME};charset=utf8mb4",
            $DB_USER,
            $DB_PASS,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
    } catch (Exception $e) {
        if (PHP_SAPI === 'cli') {
            fwrite(STDERR, "DB connection error: " . $e->getMessage() . PHP_EOL);
            exit(1);
        }
        // If web request and not installing, show error
        if (!isset($_GET['action']) || $_GET['action'] !== 'install') {
            die('Database connection error. Please check your configuration.');
        }
    }
}

if (PHP_SAPI === 'cli' && isset($argv) && count($argv) > 1) {
    if ($argv[1] === '--bg-send-profile') {
        $cid = isset($argv[2]) ? (int)$argv[2] : 0;
        $pid = isset($argv[3]) ? (int)$argv[3] : 0;
        $tmpfile = isset($argv[4]) ? $argv[4] : '';
        $recipientsText = '';
        $overrides = [];
        if ($tmpfile && is_readable($tmpfile)) {
            $raw = file_get_contents($tmpfile);
            $dec = json_decode($raw, true);
            if (is_array($dec) && isset($dec['recipients'])) {
                // support recipients as array or string
                if (is_array($dec['recipients'])) {
                    $recipientsText = implode("\n", $dec['recipients']);
                } else {
                    $recipientsText = (string)$dec['recipients'];
                }
                $overrides = isset($dec['overrides']) && is_array($dec['overrides']) ? $dec['overrides'] : [];
            } else {
                $recipientsText = $raw;
            }
        }
        try {
            $campaign = get_campaign($pdo, $cid);
            if ($campaign) {
                send_campaign_real($pdo, $campaign, $recipientsText, false, $pid, $overrides);
            }
        } catch (Exception $e) {
            // nothing to do
        }
        if ($tmpfile) @unlink($tmpfile);
        exit(0);
    }

    if ($argv[1] === '--bg-send') {
        $cid = isset($argv[2]) ? (int)$argv[2] : 0;
        $tmpfile = isset($argv[3]) ? $argv[3] : '';
        $recipientsText = '';
        $overrides = [];
        if ($tmpfile && is_readable($tmpfile)) {
            $raw = file_get_contents($tmpfile);
            $dec = json_decode($raw, true);
            if (is_array($dec) && isset($dec['recipients'])) {
                if (is_array($dec['recipients'])) {
                    $recipientsText = implode("\n", $dec['recipients']);
                } else {
                    $recipientsText = (string)$dec['recipients'];
                }
                $overrides = isset($dec['overrides']) && is_array($dec['overrides']) ? $dec['overrides'] : [];
            } else {
                $recipientsText = $raw;
            }
        }
        try {
            $campaign = get_campaign($pdo, $cid);
            if ($campaign) {
                send_campaign_real($pdo, $campaign, $recipientsText, false, null, $overrides);
            }
        } catch (Exception $e) {
            // nothing to do
        }
        if ($tmpfile) @unlink($tmpfile);
        exit(0);
    }

    if ($argv[1] === '--bg-scan-bounces') {
        try {
            process_imap_bounces($pdo);
        } catch (Exception $e) {
            // ignore
        }
        exit(0);
    }
}

session_start();

///////////////////////
//  AUTHENTICATION SYSTEM
///////////////////////
// Check if user is logged in (skip for login, logout, install, and tracking actions)
$publicActions = ['login', 'do_login', 'logout', 'install', 'do_install'];
$trackingRequests = ['t' => ['open', 'click', 'unsubscribe']];
$isPublicAction = isset($_GET['action']) && in_array($_GET['action'], $publicActions);
$isTrackingRequest = isset($_GET['t']) && in_array($_GET['t'], $trackingRequests['t']);
$isApiRequest = isset($_GET['api']);

if (!$isPublicAction && !$isTrackingRequest && !$isApiRequest && PHP_SAPI !== 'cli') {
    // Check if installation is complete
    if (!defined('INSTALLED') || !INSTALLED) {
        // Redirect to installation wizard
        if (!isset($_GET['action']) || $_GET['action'] !== 'install') {
            header('Location: ' . $_SERVER['SCRIPT_NAME'] . '?action=install');
            exit;
        }
    } elseif (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
        // Not logged in, show login page
        if (!isset($_GET['action']) || $_GET['action'] !== 'login') {
            header('Location: ' . $_SERVER['SCRIPT_NAME'] . '?action=login');
            exit;
        }
    }
}

///////////////////////
//  AUTHENTICATION HANDLERS
///////////////////////
if (isset($_GET['action']) && $_GET['action'] === 'do_login' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    if (defined('ADMIN_USERNAME') && defined('ADMIN_PASSWORD_HASH')) {
        if ($username === ADMIN_USERNAME && password_verify($password, ADMIN_PASSWORD_HASH)) {
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_username'] = $username;
            header('Location: ' . $_SERVER['SCRIPT_NAME']);
            exit;
        }
    }
    
    // Login failed
    header('Location: ' . $_SERVER['SCRIPT_NAME'] . '?action=login&error=1');
    exit;
}

if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    session_destroy();
    header('Location: ' . $_SERVER['SCRIPT_NAME'] . '?action=login&logged_out=1');
    exit;
}

// Show login page
if (isset($_GET['action']) && $_GET['action'] === 'login') {
    include_once 'login.php';
    exit;
}

// Show installation wizard
if (isset($_GET['action']) && $_GET['action'] === 'install') {
    include_once 'install_wizard.php';
    exit;
}

///////////////////////
//  OPTIONAL SCHEMA (Contacts + added columns)
///////////////////////
try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS contact_lists (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            type ENUM('global','list') NOT NULL DEFAULT 'list',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS contacts (
            id INT AUTO_INCREMENT PRIMARY KEY,
            list_id INT NOT NULL,
            email VARCHAR(255) NOT NULL,
            first_name VARCHAR(100) DEFAULT NULL,
            last_name VARCHAR(100) DEFAULT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX(list_id),
            INDEX(email)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");

    $pdo->exec("
        CREATE TABLE IF NOT EXISTS unsubscribes (
            email VARCHAR(255) PRIMARY KEY,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");

    // Ensure sending_profiles has useful columns
    try { $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS sender_name VARCHAR(255) NOT NULL DEFAULT ''"); } catch (Exception $e) {}
    // Add connection_numbers field (MaxBulk Mailer style - concurrent SMTP connections)
    try { $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS connection_numbers INT NOT NULL DEFAULT 5"); } catch (Exception $e) {}
    // Add batch_size field (number of emails to send per single SMTP connection)
    try { $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS batch_size INT NOT NULL DEFAULT 50"); } catch (Exception $e) {}

    try { $pdo->exec("ALTER TABLE campaigns ADD COLUMN IF NOT EXISTS unsubscribe_enabled TINYINT(1) NOT NULL DEFAULT 0"); } catch (Exception $e) {}
    try { $pdo->exec("ALTER TABLE campaigns ADD COLUMN IF NOT EXISTS sender_name VARCHAR(255) NOT NULL DEFAULT ''"); } catch (Exception $e) {}
    // Add campaign progress tracking fields
    try { 
        $pdo->exec("ALTER TABLE campaigns ADD COLUMN IF NOT EXISTS progress_sent INT NOT NULL DEFAULT 0");
        $pdo->exec("ALTER TABLE campaigns ADD COLUMN IF NOT EXISTS progress_total INT NOT NULL DEFAULT 0");
        $pdo->exec("ALTER TABLE campaigns ADD COLUMN IF NOT EXISTS progress_status VARCHAR(50) DEFAULT 'draft'");
    } catch (Exception $e) {}
    try {
        $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS bounce_imap_server VARCHAR(255) DEFAULT ''");
        $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS bounce_imap_user VARCHAR(255) DEFAULT ''");
        $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS bounce_imap_pass VARCHAR(255) DEFAULT ''");
    } catch (Exception $e) {}
    
    // Keep old fields for backward compatibility but they won't be shown in UI
    try { $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS send_rate INT NOT NULL DEFAULT 0"); } catch (Exception $e) {}
    try { $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS sends_used INT NOT NULL DEFAULT 0"); } catch (Exception $e) {}
    try {
        $pdo->exec("ALTER TABLE rotation_settings ADD COLUMN IF NOT EXISTS workers INT NOT NULL DEFAULT " . DEFAULT_WORKERS);
        $pdo->exec("ALTER TABLE rotation_settings ADD COLUMN IF NOT EXISTS messages_per_worker INT NOT NULL DEFAULT " . DEFAULT_MESSAGES_PER_WORKER);
    } catch (Exception $e) {}
    try {
        $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS workers INT NOT NULL DEFAULT " . DEFAULT_WORKERS);
        $pdo->exec("ALTER TABLE sending_profiles ADD COLUMN IF NOT EXISTS messages_per_worker INT NOT NULL DEFAULT " . DEFAULT_MESSAGES_PER_WORKER);
    } catch (Exception $e) {}
} catch (Exception $e) {
    // ignore creation errors
}

///////////////////////
//  HELPERS
///////////////////////
function h($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

function uuidv4()
{
    $data = random_bytes(16);
    $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
    $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}

function base64url_encode(string $data): string {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function base64url_decode(string $data): string {
    $data = strtr($data, '-_', '+/');
    return base64_decode($data);
}

function get_base_url(): string {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $script = $_SERVER['SCRIPT_NAME'] ?? '/mailer.php';
    return $scheme . '://' . $host . $script;
}

function encode_mime_header(string $str): string {
    if ($str === '') return '';
    if (preg_match('/[^\x20-\x7E]/', $str)) {
        return '=?UTF-8?B?' . base64_encode($str) . '?=';
    }
    return $str;
}

function is_unsubscribed(PDO $pdo, string $email): bool {
    if ($email === '') return false;
    try {
        $stmt = $pdo->prepare("SELECT 1 FROM unsubscribes WHERE email = ? LIMIT 1");
        $stmt->execute([strtolower($email)]);
        return (bool)$stmt->fetchColumn();
    } catch (Exception $e) {
        return false;
    }
}

function add_to_unsubscribes(PDO $pdo, string $email) {
    if ($email === '') return;
    try {
        $stmt = $pdo->prepare("INSERT IGNORE INTO unsubscribes (email) VALUES (?)");
        $stmt->execute([strtolower($email)]);
    } catch (Exception $e) {
        // ignore
    }
}

/**
 * Try to execute a background command using several strategies.
 * Returns true if a background launch was attempted and likely succeeded.
 */
function try_background_exec(string $cmd): bool {
    $cmd = trim($cmd);
    if ($cmd === '') return false;

    // Windows: try start /B
    if (stripos(PHP_OS, 'WIN') === 0) {
        // Note: Windows background execution is less reliable here; we attempt it.
        $winCmd = "start /B " . $cmd;
        @pclose(@popen($winCmd, "r"));
        return true;
    }

    // Prefer proc_open if available
    if (function_exists('proc_open')) {
        $descriptors = [
            0 => ["pipe", "r"],
            1 => ["file", "/dev/null", "a"],
            2 => ["file", "/dev/null", "a"],
        ];
        $process = @proc_open($cmd, $descriptors, $pipes);
        if (is_resource($process)) {
            @proc_close($process);
            return true;
        }
    }

    // Try exec with async redirection
    if (function_exists('exec')) {
        @exec($cmd . " > /dev/null 2>&1 &");
        return true;
    }

    // Try shell_exec
    if (function_exists('shell_exec')) {
        @shell_exec($cmd . " > /dev/null 2>&1 &");
        return true;
    }

    // Try popen
    if (function_exists('popen')) {
        @pclose(@popen($cmd . " > /dev/null 2>&1 &", "r"));
        return true;
    }

    // No method available
    return false;
}

/**
 * Spawn a detached background PHP process that runs this script with --bg-send
 * Returns true if spawn was likely successful.
 */
function spawn_background_send(PDO $pdo, int $campaignId, string $recipientsText, array $overrides = []): bool {
    $tmpDir = sys_get_temp_dir();
    $tmpFile = tempnam($tmpDir, 'ss_send_');
    if ($tmpFile === false) return false;
    $payload = ['recipients' => $recipientsText, 'overrides' => $overrides];
    if (file_put_contents($tmpFile, json_encode($payload)) === false) {
        @unlink($tmpFile);
        return false;
    }

    $php = PHP_BINARY ?: 'php';
    $script = $_SERVER['SCRIPT_FILENAME'] ?? __FILE__;
    // Build command for UNIX-like systems; quoted args
    $cmd = escapeshellcmd($php) . ' -f ' . escapeshellarg($script)
         . ' -- ' . '--bg-send ' . escapeshellarg((string)$campaignId) . ' ' . escapeshellarg($tmpFile);

    // Try variants: with nohup, with & redirect, etc.
    $candidates = [
        $cmd . " > /dev/null 2>&1 &",
        "nohup " . $cmd . " > /dev/null 2>&1 &",
        $cmd, // fallback
    ];

    foreach ($candidates as $c) {
        if (try_background_exec($c)) {
            // the CLI worker will remove the tmpfile after processing; but if spawn failed and we decide to fallback
            return true;
        }
    }

    // if none worked, cleanup and return false (caller will fallback to synchronous send)
    @unlink($tmpFile);
    return false;
}

/**
 * Spawn a detached background PHP process that runs this script with --bg-send-profile <campaignId> <profileId> <tmpfile>
 * Returns true if spawn was likely successful.
 *
 * Accepts $overrides array which will be written to the tmpfile JSON so the CLI worker can apply runtime overrides
 * such as 'send_rate' (messages/sec).
 */
function spawn_background_send_profile(PDO $pdo, int $campaignId, int $profileId, string $recipientsText, array $overrides = []): bool {
    $tmpDir = sys_get_temp_dir();
    $tmpFile = tempnam($tmpDir, 'ss_send_pf_');
    if ($tmpFile === false) return false;
    $payload = ['recipients' => $recipientsText, 'overrides' => $overrides];
    if (file_put_contents($tmpFile, json_encode($payload)) === false) {
        @unlink($tmpFile);
        return false;
    }

    $php = PHP_BINARY ?: 'php';
    $script = $_SERVER['SCRIPT_FILENAME'] ?? __FILE__;
    $cmd = escapeshellcmd($php) . ' -f ' . escapeshellarg($script)
         . ' -- ' . '--bg-send-profile ' . escapeshellarg((string)$campaignId) . ' ' . escapeshellarg((string)$profileId) . ' ' . escapeshellarg($tmpFile);

    $candidates = [
        $cmd . " > /dev/null 2>&1 &",
        "nohup " . $cmd . " > /dev/null 2>&1 &",
        $cmd,
    ];

    foreach ($candidates as $c) {
        if (try_background_exec($c)) {
            return true;
        }
    }

    @unlink($tmpFile);
    return false;
}

/**
 * Spawn multiple parallel workers for sending emails
 * Splits recipients into chunks and spawns a worker for each chunk
 */
function spawn_parallel_workers(PDO $pdo, int $campaignId, array $recipients, int $workers, int $messagesPerWorker, ?int $profileId = null, array $overrides = []): array {
    $totalRecipients = count($recipients);
    if ($totalRecipients === 0) return ['success' => true, 'workers' => 0];
    
    // Validate workers parameter
    $workers = max(MIN_WORKERS, min(MAX_WORKERS, $workers));
    $messagesPerWorker = max(MIN_MESSAGES_PER_WORKER, $messagesPerWorker);
    
    // Calculate chunk size based on messages_per_worker
    $chunkSize = $messagesPerWorker;
    
    // Split recipients into chunks
    $chunks = array_chunk($recipients, $chunkSize);
    $totalChunks = count($chunks);
    
    // Distribute chunks across workers - each worker may process multiple chunks
    $spawnedWorkers = 0;
    $failures = [];
    
    // If we have more chunks than workers, we need to combine chunks for some workers
    if ($totalChunks > $workers) {
        // Calculate how many chunks each worker should handle
        $chunksPerWorker = (int)ceil($totalChunks / $workers);
        
        for ($workerIdx = 0; $workerIdx < $workers; $workerIdx++) {
            $startChunk = $workerIdx * $chunksPerWorker;
            $endChunk = min($startChunk + $chunksPerWorker, $totalChunks);
            
            // Combine multiple chunks for this worker
            $workerRecipients = [];
            for ($chunkIdx = $startChunk; $chunkIdx < $endChunk; $chunkIdx++) {
                if (isset($chunks[$chunkIdx])) {
                    $workerRecipients = array_merge($workerRecipients, $chunks[$chunkIdx]);
                }
            }
            
            if (empty($workerRecipients)) continue;
            
            $chunkText = implode("\n", $workerRecipients);
            
            if ($profileId !== null) {
                $spawned = spawn_background_send_profile($pdo, $campaignId, $profileId, $chunkText, $overrides);
            } else {
                $spawned = spawn_background_send($pdo, $campaignId, $chunkText, $overrides);
            }
            
            if ($spawned) {
                $spawnedWorkers++;
            } else {
                $failures[] = ['chunk' => $workerRecipients, 'profileId' => $profileId, 'overrides' => $overrides];
            }
        }
    } else {
        // We have fewer chunks than workers, spawn one worker per chunk
        foreach ($chunks as $idx => $chunk) {
            $chunkText = implode("\n", $chunk);
            
            if ($profileId !== null) {
                $spawned = spawn_background_send_profile($pdo, $campaignId, $profileId, $chunkText, $overrides);
            } else {
                $spawned = spawn_background_send($pdo, $campaignId, $chunkText, $overrides);
            }
            
            if ($spawned) {
                $spawnedWorkers++;
            } else {
                $failures[] = ['chunk' => $chunk, 'profileId' => $profileId, 'overrides' => $overrides];
            }
        }
    }
    
    return [
        'success' => empty($failures),
        'workers' => $spawnedWorkers,
        'failures' => $failures
    ];
}

/**
 * Spawn a detached background PHP process to scan IMAP bounce mailboxes
 */
function spawn_bounce_scan(PDO $pdo): bool {
    $php = PHP_BINARY ?: 'php';
    $script = $_SERVER['SCRIPT_FILENAME'] ?? __FILE__;
    $cmd = escapeshellcmd($php) . ' -f ' . escapeshellarg($script)
         . ' -- ' . '--bg-scan-bounces';

    $candidates = [
        $cmd . " > /dev/null 2>&1 &",
        "nohup " . $cmd . " > /dev/null 2>&1 &",
        $cmd,
    ];

    foreach ($candidates as $c) {
        if (try_background_exec($c)) {
            return true;
        }
    }
    return false;
}

/**
 * Inject open & click tracking inside HTML
 */
function build_tracked_html(array $campaign, string $recipientEmail = ''): string
{
    $html = $campaign['html'] ?? '';
    if ($html === '') {
        return '';
    }

    $cid  = (int)$campaign['id'];
    $base = get_base_url();

    $rParam = '';
    if ($recipientEmail !== '') {
        $rParam = '&r=' . rawurlencode(base64url_encode(strtolower($recipientEmail)));
    }

    $unsubscribeEnabled = !empty($campaign['unsubscribe_enabled']) ? true : false;

    $openUrl = $base . '?t=open&cid=' . $cid . $rParam;
    $pixel   = '<img src="' . $openUrl . '" alt="" width="1" height="1" style="display:none!important;max-height:0;overflow:hidden;">';

    if (stripos($html, '</body>') !== false) {
        $html = preg_replace('~</body>~i', $pixel . '</body>', $html, 1);
    } else {
        $html .= $pixel;
    }

    $pattern_unsub = '~<a\b([^>]*?)\bhref\s*=\s*(["\'])(.*?)\2([^>]*)>(.*?)</a>~is';
    $foundUnsubAnchor = false;
    $html = preg_replace_callback($pattern_unsub, function ($m) use ($base, $cid, $rParam, $unsubscribeEnabled, &$foundUnsubAnchor) {
        $beforeAttrs = $m[1];
        $quote       = $m[2];
        $href        = $m[3];
        $afterAttrs  = $m[4];
        $innerText   = $m[5];

        if (stripos($innerText, 'unsubscribe') !== false) {
            if (!$unsubscribeEnabled) {
                return '';
            }
            $foundUnsubAnchor = true;
            $unsubUrl = $base . '?t=unsubscribe&cid=' . $cid . $rParam;
            return '<a' . $beforeAttrs . ' href=' . $quote . $unsubUrl . $quote . $afterAttrs . '>' . $innerText . '</a>';
        }
        return $m[0];
    }, $html);

    if ($unsubscribeEnabled && !$foundUnsubAnchor) {
        $unsubUrl = $base . '?t=unsubscribe&cid=' . $cid . $rParam;
        $unsubBlock = '<div style="text-align:center;margin-top:18px;color:#777;font-size:13px;">'
                     . '<a href="' . $unsubUrl . '" style="color:#1A82E2;">Unsubscribe</a>'
                     . '</div>';
        if (stripos($html, '</body>') !== false) {
            $html = preg_replace('~</body>~i', $unsubBlock . '</body>', $html, 1);
        } else {
            $html .= $unsubBlock;
        }
    }

    $pattern = '~<a\b([^>]*?)\bhref\s*=\s*(["\'])(.*?)\2([^>]*)>~i';

    $html = preg_replace_callback($pattern, function ($m) use ($base, $cid, $rParam) {
        $beforeAttrs = $m[1];
        $quote       = $m[2];
        $href        = $m[3];
        $afterAttrs  = $m[4];

        $trimHref = trim($href);

        if (
            $trimHref === '' ||
            stripos($trimHref, 'mailto:') === 0 ||
            stripos($trimHref, 'javascript:') === 0 ||
            $trimHref[0] === '#'
        ) {
            return $m[0];
        }

        if (stripos($trimHref, '?t=click') !== false && stripos($trimHref, 'cid=') !== false) {
            return $m[0];
        }
        if (stripos($trimHref, '?t=unsubscribe') !== false && stripos($trimHref, 'cid=') !== false) {
            return $m[0];
        }

        $encodedUrl = base64url_encode($trimHref);
        $trackUrl   = $base . '?t=click&cid=' . $cid . '&u=' . rawurlencode($encodedUrl) . $rParam;

        return '<a' . $beforeAttrs . ' href=' . $quote . $trackUrl . $quote . $afterAttrs . '>';
    }, $html);

    return $html;
}

function smtp_check_connection(array $profile): array {
    $log = [];

    $host = trim($profile['host'] ?? '');
    $port = (int)($profile['port'] ?? 587);
    $user = trim($profile['username'] ?? '');
    $pass = (string)($profile['password'] ?? '');

    if ($host === '') {
        return ['ok'=>false,'msg'=>'Missing SMTP host','log'=>$log];
    }

    $remoteHost = ($port === 465 ? "ssl://{$host}" : $host);

    $errno = 0;
    $errstr = '';
    $socket = @fsockopen($remoteHost, $port, $errno, $errstr, 10);
    if (!$socket) {
        $log[] = "connect_error: {$errno} {$errstr}";
        return ['ok'=>false,'msg'=>"SMTP connect error: {$errno} {$errstr}", 'log'=>$log];
    }
    stream_set_timeout($socket, 10);

    $read = function() use ($socket, &$log) {
        $data = '';
        while ($str = fgets($socket, 515)) {
            $data .= $str;
            $log[] = rtrim($str, "\r\n");
            if (strlen($str) < 4) break;
            if (substr($str, 3, 1) === ' ') break;
        }
        $code = isset($data[0]) ? substr($data, 0, 3) : null;
        $msg  = isset($data[4]) ? trim(substr($data, 4)) : trim($data);
        return [$code, $msg, $data];
    };
    $write = function(string $cmd) use ($socket, $read, &$log) {
        fputs($socket, $cmd . "\r\n");
        $log[] = 'C: ' . $cmd;
        return $read();
    };

    list($gcode, $gmsg) = $read();
    if (!is_string($gcode) || substr($gcode,0,1) !== '2') {
        fclose($socket);
        $log[] = 'banner_failed';
        return ['ok'=>false,'msg'=>"SMTP banner failed: {$gcode} {$gmsg}", 'log'=>$log];
    }

    list($ecode, $emsg) = $write('EHLO localhost');
    if (!is_string($ecode) || substr($ecode,0,1) !== '2') {
        list($hcode, $hmsg) = $write('HELO localhost');
        if (!is_string($hcode) || substr($hcode,0,1) !== '2') {
            fclose($socket);
            $log[] = 'ehlo_failed';
            return ['ok'=>false,'msg'=>"EHLO/HELO failed: {$ecode} / {$hcode}", 'log'=>$log];
        } else {
            $ecode = $hcode; $emsg = $hmsg;
        }
    }

    if ($user !== '') {
        list($acode, $amsg) = $write('AUTH LOGIN');
        if (!is_string($acode) || substr($acode,0,1) !== '3') {
            fclose($socket);
            return ['ok'=>true,'msg'=>'Connected; AUTH not required/accepted','log'=>$log];
        }
        list($ucode, $umsg) = $write(base64_encode($user));
        if (!is_string($ucode) || substr($ucode,0,1) !== '3') {
            fclose($socket);
            return ['ok'=>false,'msg'=>"AUTH username rejected: {$ucode} {$umsg}", 'log'=>$log];
        }
        list($pcode, $pmsg) = $write(base64_encode($pass));
        if (!is_string($pcode) || substr($pcode,0,1) !== '2') {
            fclose($socket);
            return ['ok'=>false,'msg'=>"AUTH password rejected: {$pcode} {$pmsg}", 'log'=>$log];
        }
    }

    $write('QUIT');
    fclose($socket);

    return ['ok'=>true,'msg'=>'SMTP connect OK','log'=>$log];
}

function api_check_connection(array $profile): array {
    $apiUrl = trim($profile['api_url'] ?? '');
    $apiKey = trim($profile['api_key'] ?? '');

    $log = [];

    if ($apiUrl === '') {
        return ['ok'=>false,'msg'=>'Missing API URL','log'=>$log];
    }

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_NOBODY, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $headers = [];
    if ($apiKey !== '') {
        $headers[] = 'Authorization: Bearer ' . $apiKey;
    }
    if (!empty($profile['headers_json'])) {
        $extra = json_decode($profile['headers_json'], true);
        if (is_array($extra)) {
            foreach ($extra as $k => $v) $headers[] = "{$k}: {$v}";
        }
    }
    if (!empty($headers)) curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

    $resp = curl_exec($ch);
    $err = null;
    if ($resp === false) {
        $err = curl_error($ch);
        curl_setopt($ch, CURLOPT_NOBODY, false);
        curl_setopt($ch, CURLOPT_HTTPGET, true);
        curl_setopt($ch, CURLOPT_POST, false);
        $resp2 = curl_exec($ch);
        if ($resp2 === false) {
            $err2 = curl_error($ch);
            curl_close($ch);
            return ['ok'=>false,'msg'=>"cURL error: {$err} / {$err2}", 'log'=>[$err, $err2]];
        }
    }
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $log[] = 'HTTP CODE: ' . $code;
    if ($code >= 200 && $code < 400) {
        return ['ok'=>true,'msg'=>'API reachable (HTTP ' . $code . ')','log'=>$log];
    } else {
        return ['ok'=>false,'msg'=>'API returned HTTP ' . $code,'log'=>$log];
    }
}

/**
 * SMTP send (returns structured array)
 */
function smtp_send_mail(array $profile, array $campaign, string $to, string $html): array
{
    $log = [];

    $host = trim($profile['host'] ?? '');
    $port = (int)($profile['port'] ?? 587);
    $user = trim($profile['username'] ?? '');
    $pass = (string)($profile['password'] ?? '');

    $from = trim($campaign['from_email'] ?? '');

    if ($host === '' || $from === '' || $to === '') {
        return [
            'ok' => false,
            'type' => 'bounce',
            'code' => null,
            'msg' => 'SMTP: missing host/from/to',
            'stage' => 'connect',
            'log' => $log,
        ];
    }

    $remoteHost = ($port === 465 ? "ssl://{$host}" : $host);

    $errno = 0;
    $errstr = '';
    $socket = @fsockopen($remoteHost, $port, $errno, $errstr, 25);
    if (!$socket) {
        return [
            'ok' => false,
            'type' => 'bounce',
            'code' => null,
            'msg' => "SMTP connect error: {$errno} {$errstr}",
            'stage' => 'connect',
            'log' => $log,
        ];
    }
    stream_set_timeout($socket, 25);

    $read = function() use ($socket, &$log) {
        $data = '';
        while ($str = fgets($socket, 515)) {
            $data .= $str;
            $log[] = rtrim($str, "\r\n");
            if (strlen($str) < 4) break;
            if (substr($str, 3, 1) === ' ') break;
        }
        $code = isset($data[0]) ? substr($data, 0, 3) : null;
        $msg  = isset($data[4]) ? trim(substr($data, 4)) : trim($data);
        return [$code, $msg, $data];
    };
    $write = function(string $cmd) use ($socket, $read, &$log) {
        fputs($socket, $cmd . "\r\n");
        $log[] = 'C: ' . $cmd;
        return $read();
    };

    list($gcode, $gmsg) = $read();
    if (!is_string($gcode) || substr($gcode,0,1) !== '2') {
        fclose($socket);
        return [
            'ok' => false,
            'type' => 'bounce',
            'code' => $gcode,
            'msg' => $gmsg,
            'stage' => 'banner',
            'log' => $log,
        ];
    }

    list($ecode, $emsg) = $write('EHLO localhost');
    if (!is_string($ecode) || substr($ecode,0,1) !== '2') {
        list($hcode, $hmsg) = $write('HELO localhost');
        if (!is_string($hcode) || substr($hcode,0,1) !== '2') {
            fclose($socket);
            return [
                'ok' => false,
                'type' => 'bounce',
                'code' => $ecode ?: $hcode,
                'msg' => $emsg ?: $hmsg,
                'stage' => 'ehlo',
                'log' => $log,
            ];
        } else {
            $ecode = $hcode; $emsg = $hmsg;
        }
    }

    if ($port !== 465 && is_string($emsg) && stripos(implode("\n", $log), 'STARTTLS') !== false) {
        list($tcode, $tmsg) = $write('STARTTLS');
        if (!is_string($tcode) || substr($tcode, 0, 3) !== '220') {
            fclose($socket);
            return [
                'ok' => false,
                'type' => 'bounce',
                'code' => $tcode,
                'msg' => $tmsg,
                'stage' => 'starttls',
                'log' => $log,
            ];
        }
        $cryptoMethod = defined('STREAM_CRYPTO_METHOD_TLS_CLIENT')
            ? STREAM_CRYPTO_METHOD_TLS_CLIENT
            : (STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT
               | STREAM_CRYPTO_METHOD_TLSv1_1_CLIENT
               | STREAM_CRYPTO_METHOD_TLSv1_0_CLIENT);

        if (!stream_socket_enable_crypto($socket, true, $cryptoMethod)) {
            fclose($socket);
            return [
                'ok' => false,
                'type' => 'bounce',
                'code' => null,
                'msg' => 'Unable to start TLS crypto',
                'stage' => 'starttls',
                'log' => $log,
            ];
        }
        list($ecode, $emsg) = $write('EHLO localhost');
    }

    if ($user !== '') {
        list($acode, $amsg) = $write('AUTH LOGIN');
        if (!is_string($acode) || substr($acode,0,1) !== '3') {
            fclose($socket);
            return ['ok'=>false, 'type'=>'bounce', 'code'=>$acode, 'msg'=>$amsg, 'stage'=>'auth', 'log'=>$log];
        }
        list($ucode, $umsg) = $write(base64_encode($user));
        if (!is_string($ucode) || substr($ucode,0,1) !== '3') {
            fclose($socket);
            return ['ok'=>false, 'type'=>'bounce', 'code'=>$ucode, 'msg'=>$umsg, 'stage'=>'auth_user', 'log'=>$log];
        }
        list($pcode, $pmsg) = $write(base64_encode($pass));
        if (!is_string($pcode) || substr($pcode,0,1) !== '2') {
            fclose($socket);
            return ['ok'=>false, 'type'=>'bounce', 'code'=>$pcode, 'msg'=>$pmsg, 'stage'=>'auth_pass', 'log'=>$log];
        }
    }

    list($mcode, $mmsg) = $write('MAIL FROM: <' . $from . '>');
    if (!is_string($mcode)) $mcode = null;
    if ($mcode !== null && $mcode !== '' && $mcode[0] === '5') {
        fclose($socket);
        return ['ok'=>false,'type'=>'bounce','code'=>$mcode,'msg'=>$mmsg,'stage'=>'mail_from','log'=>$log];
    } elseif ($mcode !== null && $mcode !== '' && $mcode[0] === '4') {
        fclose($socket);
        return ['ok'=>false,'type'=>'deferred','code'=>$mcode,'msg'=>$mmsg,'stage'=>'mail_from','log'=>$log];
    } elseif ($mcode === null) {
        fclose($socket);
        return ['ok'=>false,'type'=>'bounce','code'=>null,'msg'=>'MAIL FROM unknown response','stage'=>'mail_from','log'=>$log];
    }

    list($rcode, $rmsg) = $write('RCPT TO: <' . $to . '>');
    if (!is_string($rcode)) $rcode = null;
    if ($rcode !== null && $rcode[0] === '5') {
        fclose($socket);
        return ['ok'=>false,'type'=>'bounce','code'=>$rcode,'msg'=>$rmsg,'stage'=>'rcpt_to','log'=>$log];
    } elseif ($rcode !== null && $rcode[0] === '4') {
        fclose($socket);
        return ['ok'=>false,'type'=>'deferred','code'=>$rcode,'msg'=>$rmsg,'stage'=>'rcpt_to','log'=>$log];
    } elseif ($rcode === null) {
        fclose($socket);
        return ['ok'=>false,'type'=>'bounce','code'=>null,'msg'=>'RCPT TO unknown response','stage'=>'rcpt_to','log'=>$log];
    }

    list($dcode, $dmsg) = $write('DATA');
    if (!is_string($dcode) || substr($dcode,0,1) !== '3') {
        fclose($socket);
        return ['ok'=>false,'type'=>'bounce','code'=>$dcode,'msg'=>$dmsg,'stage'=>'data_cmd','log'=>$log];
    }

    $subject = encode_mime_header($campaign['subject'] ?? '');

    $headers = '';
    $headers .= "Date: " . gmdate('D, d M Y H:i:s T') . "\r\n";

    $fromDisplay = '';
    if (!empty($campaign['sender_name'])) {
        $fromDisplay = encode_mime_header($campaign['sender_name']) . " <{$from}>";
    } else {
        $fromDisplay = "<{$from}>";
    }
    $headers .= "From: {$fromDisplay}\r\n";

    $headers .= "To: <{$to}>\r\n";
    if ($subject !== '') {
        $headers .= "Subject: {$subject}\r\n";
    }

    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "Content-Transfer-Encoding: base64\r\n\r\n";

    $body = chunk_split(base64_encode($html), 76, "\r\n");

    $message = $headers . $body . "\r\n.\r\n";
    fputs($socket, $message);
    $log[] = 'C: <message data...>';

    list($finalCode, $finalMsg) = $read();

    if (!is_string($finalCode) || $finalCode[0] !== '2') {
        fclose($socket);
        $type = 'bounce';
        if (is_string($finalCode) && $finalCode[0] === '4') $type = 'deferred';
        return ['ok'=>false,'type'=>$type,'code'=>$finalCode,'msg'=>$finalMsg,'stage'=>'data_end','log'=>$log];
    }

    $write('QUIT');
    fclose($socket);

    return [
        'ok'    => true,
        'type'  => 'delivered',
        'code'  => $finalCode,
        'msg'   => $finalMsg,
        'stage' => 'done',
        'log'   => $log,
    ];
}

/**
 * Send multiple emails using a single SMTP connection (batch sending)
 * This improves performance by reusing the connection instead of opening/closing for each email
 * 
 * @param array $profile SMTP profile configuration
 * @param array $campaign Campaign data
 * @param array $recipients Array of recipient email addresses
 * @param array $htmlMap Optional map of recipient => custom HTML (if not provided, same HTML for all)
 * @return array Results with 'results' array containing per-recipient results
 */
function smtp_send_batch(array $profile, array $campaign, array $recipients, array $htmlMap = []): array
{
    $log = [];
    $results = [];
    
    if (empty($recipients)) {
        return [
            'ok' => false,
            'error' => 'No recipients provided',
            'log' => $log,
            'results' => []
        ];
    }

    $host = trim($profile['host'] ?? '');
    $port = (int)($profile['port'] ?? 587);
    $user = trim($profile['username'] ?? '');
    $pass = (string)($profile['password'] ?? '');
    $from = trim($campaign['from_email'] ?? '');

    if ($host === '' || $from === '') {
        return [
            'ok' => false,
            'error' => 'Missing SMTP host or from address',
            'log' => $log,
            'results' => []
        ];
    }

    $remoteHost = ($port === 465 ? "ssl://{$host}" : $host);
    
    // Open connection
    $errno = 0;
    $errstr = '';
    $socket = @fsockopen($remoteHost, $port, $errno, $errstr, 25);
    if (!$socket) {
        $error = "SMTP connect error: {$errno} {$errstr}";
        $log[] = $error;
        return [
            'ok' => false,
            'error' => $error,
            'log' => $log,
            'results' => []
        ];
    }
    stream_set_timeout($socket, 25);

    $read = function() use ($socket, &$log) {
        $data = '';
        while ($str = fgets($socket, 515)) {
            $data .= $str;
            $log[] = rtrim($str, "\r\n");
            if (strlen($str) < 4) break;
            if (substr($str, 3, 1) === ' ') break;
        }
        $code = isset($data[0]) ? substr($data, 0, 3) : null;
        $msg  = isset($data[4]) ? trim(substr($data, 4)) : trim($data);
        return [$code, $msg, $data];
    };
    
    $write = function(string $cmd) use ($socket, $read, &$log) {
        fputs($socket, $cmd . "\r\n");
        $log[] = 'C: ' . $cmd;
        return $read();
    };

    // Banner
    list($gcode, $gmsg) = $read();
    if (!is_string($gcode) || substr($gcode,0,1) !== '2') {
        fclose($socket);
        return [
            'ok' => false,
            'error' => "SMTP banner failed: {$gcode} {$gmsg}",
            'log' => $log,
            'results' => []
        ];
    }

    // EHLO/HELO
    list($ecode, $emsg) = $write('EHLO localhost');
    if (!is_string($ecode) || substr($ecode,0,1) !== '2') {
        list($hcode, $hmsg) = $write('HELO localhost');
        if (!is_string($hcode) || substr($hcode,0,1) !== '2') {
            fclose($socket);
            return [
                'ok' => false,
                'error' => "EHLO/HELO failed: {$ecode} / {$hcode}",
                'log' => $log,
                'results' => []
            ];
        } else {
            $ecode = $hcode; $emsg = $hmsg;
        }
    }

    // STARTTLS if needed
    if ($port !== 465 && is_string($emsg) && stripos(implode("\n", $log), 'STARTTLS') !== false) {
        list($tcode, $tmsg) = $write('STARTTLS');
        if (!is_string($tcode) || substr($tcode, 0, 3) !== '220') {
            fclose($socket);
            return [
                'ok' => false,
                'error' => "STARTTLS failed: {$tcode} {$tmsg}",
                'log' => $log,
                'results' => []
            ];
        }
        $cryptoMethod = defined('STREAM_CRYPTO_METHOD_TLS_CLIENT')
            ? STREAM_CRYPTO_METHOD_TLS_CLIENT
            : (STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT
               | STREAM_CRYPTO_METHOD_TLSv1_1_CLIENT
               | STREAM_CRYPTO_METHOD_TLSv1_0_CLIENT);

        if (!stream_socket_enable_crypto($socket, true, $cryptoMethod)) {
            fclose($socket);
            return [
                'ok' => false,
                'error' => 'Unable to start TLS crypto',
                'log' => $log,
                'results' => []
            ];
        }
        list($ecode, $emsg) = $write('EHLO localhost');
    }

    // AUTH
    if ($user !== '') {
        list($acode, $amsg) = $write('AUTH LOGIN');
        if (!is_string($acode) || substr($acode,0,1) !== '3') {
            fclose($socket);
            return [
                'ok' => false,
                'error' => "AUTH LOGIN failed: {$acode} {$amsg}",
                'log' => $log,
                'results' => []
            ];
        }
        list($ucode, $umsg) = $write(base64_encode($user));
        if (!is_string($ucode) || substr($ucode,0,1) !== '3') {
            fclose($socket);
            return [
                'ok' => false,
                'error' => "AUTH username rejected: {$ucode} {$umsg}",
                'log' => $log,
                'results' => []
            ];
        }
        list($pcode, $pmsg) = $write(base64_encode($pass));
        if (!is_string($pcode) || substr($pcode,0,1) !== '2') {
            fclose($socket);
            return [
                'ok' => false,
                'error' => "AUTH password rejected: {$pcode} {$pmsg}",
                'log' => $log,
                'results' => []
            ];
        }
    }

    // Now send each email using the same connection
    foreach ($recipients as $to) {
        $to = trim($to);
        if ($to === '') continue;
        
        // Determine HTML for this recipient
        $html = isset($htmlMap[$to]) ? $htmlMap[$to] : (isset($htmlMap['default']) ? $htmlMap['default'] : '');

        // MAIL FROM
        list($mcode, $mmsg) = $write('MAIL FROM: <' . $from . '>');
        if (!is_string($mcode)) $mcode = null;
        if ($mcode !== null && $mcode !== '' && $mcode[0] === '5') {
            $results[$to] = [
                'ok' => false,
                'type' => 'bounce',
                'code' => $mcode,
                'msg' => $mmsg,
                'stage' => 'mail_from',
                'log' => []
            ];
            // Try to continue with next recipient (some servers allow this)
            continue;
        } elseif ($mcode !== null && $mcode !== '' && $mcode[0] === '4') {
            $results[$to] = [
                'ok' => false,
                'type' => 'deferred',
                'code' => $mcode,
                'msg' => $mmsg,
                'stage' => 'mail_from',
                'log' => []
            ];
            continue;
        } elseif ($mcode === null) {
            $results[$to] = [
                'ok' => false,
                'type' => 'bounce',
                'code' => null,
                'msg' => 'MAIL FROM unknown response',
                'stage' => 'mail_from',
                'log' => []
            ];
            continue;
        }

        // RCPT TO
        list($rcode, $rmsg) = $write('RCPT TO: <' . $to . '>');
        if (!is_string($rcode)) $rcode = null;
        if ($rcode !== null && $rcode[0] === '5') {
            $results[$to] = [
                'ok' => false,
                'type' => 'bounce',
                'code' => $rcode,
                'msg' => $rmsg,
                'stage' => 'rcpt_to',
                'log' => []
            ];
            continue;
        } elseif ($rcode !== null && $rcode[0] === '4') {
            $results[$to] = [
                'ok' => false,
                'type' => 'deferred',
                'code' => $rcode,
                'msg' => $rmsg,
                'stage' => 'rcpt_to',
                'log' => []
            ];
            continue;
        } elseif ($rcode === null) {
            $results[$to] = [
                'ok' => false,
                'type' => 'bounce',
                'code' => null,
                'msg' => 'RCPT TO unknown response',
                'stage' => 'rcpt_to',
                'log' => []
            ];
            continue;
        }

        // DATA command
        list($dcode, $dmsg) = $write('DATA');
        if (!is_string($dcode) || substr($dcode,0,1) !== '3') {
            $results[$to] = [
                'ok' => false,
                'type' => 'bounce',
                'code' => $dcode,
                'msg' => $dmsg,
                'stage' => 'data_cmd',
                'log' => []
            ];
            continue;
        }

        // Build message
        $subject = encode_mime_header($campaign['subject'] ?? '');
        $headers = '';
        $headers .= "Date: " . gmdate('D, d M Y H:i:s T') . "\r\n";

        $fromDisplay = '';
        if (!empty($campaign['sender_name'])) {
            $fromDisplay = encode_mime_header($campaign['sender_name']) . " <{$from}>";
        } else {
            $fromDisplay = "<{$from}>";
        }
        $headers .= "From: {$fromDisplay}\r\n";
        $headers .= "To: <{$to}>\r\n";
        if ($subject !== '') {
            $headers .= "Subject: {$subject}\r\n";
        }
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= "Content-Transfer-Encoding: base64\r\n\r\n";

        $body = chunk_split(base64_encode($html), 76, "\r\n");
        
        // Send message data
        fputs($socket, $headers . $body);
        // Send termination sequence (CRLF.CRLF)
        fputs($socket, ".\r\n");
        $log[] = 'C: <message data...>';

        list($finalCode, $finalMsg) = $read();

        if (!is_string($finalCode) || $finalCode[0] !== '2') {
            $type = 'bounce';
            if (is_string($finalCode) && $finalCode[0] === '4') $type = 'deferred';
            $results[$to] = [
                'ok' => false,
                'type' => $type,
                'code' => $finalCode,
                'msg' => $finalMsg,
                'stage' => 'data_end',
                'log' => []
            ];
        } else {
            $results[$to] = [
                'ok' => true,
                'type' => 'delivered',
                'code' => $finalCode,
                'msg' => $finalMsg,
                'stage' => 'done',
                'log' => []
            ];
        }
    }

    // Close connection
    $write('QUIT');
    fclose($socket);

    // Check overall success
    $successCount = 0;
    foreach ($results as $result) {
        if (!empty($result['ok'])) $successCount++;
    }

    return [
        'ok' => $successCount > 0,
        'total' => count($recipients),
        'success' => $successCount,
        'failed' => count($recipients) - $successCount,
        'log' => $log,
        'results' => $results
    ];
}

function api_send_mail(array $profile, array $campaign, string $to, string $html): array
{
    $apiUrl = trim($profile['api_url'] ?? '');
    $apiKey = trim($profile['api_key'] ?? '');
    $from   = trim($campaign['from_email'] ?? '');

    $log = [];

    if ($apiUrl === '' || $from === '' || $to === '') {
        return [
            'ok' => false,
            'type' => 'bounce',
            'code' => null,
            'msg' => 'API: missing api_url/from/to',
            'stage' => 'api',
            'log' => $log,
        ];
    }

    $headers = [
        'Content-Type: application/json',
    ];
    if ($apiKey !== '') {
        $headers[] = 'Authorization: Bearer ' . $apiKey;
    }

    if (!empty($profile['headers_json'])) {
        $extra = json_decode($profile['headers_json'], true);
        if (is_array($extra)) {
            foreach ($extra as $k => $v) {
                $headers[] = $k . ': ' . $v;
            }
        }
    }

    $payload = [
        'to'      => $to,
        'from'    => $from,
        'subject' => $campaign['subject'] ?? '',
        'html'    => $html,
        'from_name' => $campaign['sender_name'] ?? '',
    ];

    $log[] = 'API POST ' . $apiUrl . ' PAYLOAD ' . json_encode($payload);

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, 25);

    $resp = curl_exec($ch);
    if ($resp === false) {
        $err = curl_error($ch);
        curl_close($ch);
        $log[] = 'CURL ERROR: ' . $err;
        return [
            'ok' => false,
            'type' => 'bounce',
            'code' => null,
            'msg' => 'API cURL error: ' . $err,
            'stage' => 'api',
            'log' => $log,
        ];
    }

    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    $log[] = 'API RESPONSE CODE: ' . $code . ' BODY: ' . substr($resp,0,1000);

    if ($code >= 200 && $code < 300) {
        return [
            'ok' => true,
            'type' => 'delivered',
            'code' => (string)$code,
            'msg'  => substr($resp,0,1000),
            'stage'=> 'api',
            'log'  => $log,
        ];
    }

    $etype = ($code >= 500) ? 'bounce' : 'deferred';
    return [
        'ok' => false,
        'type' => $etype,
        'code' => (string)$code,
        'msg'  => substr($resp,0,1000),
        'stage'=> 'api',
        'log'  => $log,
    ];
}

function get_campaign(PDO $pdo, int $id) {
    $stmt = $pdo->prepare("SELECT * FROM campaigns WHERE id = ?");
    $stmt->execute([$id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) return false;
    if (isset($row['html']) && is_string($row['html'])) {
        if (strpos($row['html'], 'BASE64:') === 0) {
            $decoded = base64_decode(substr($row['html'], 7));
            if ($decoded !== false) {
                $row['html'] = $decoded;
            }
        }
    }
    return $row;
}

function get_campaign_stats(PDO $pdo, int $id) {
    $stats = [
        'delivered'   => 0,
        'delivered_raw' => 0,
        'open'        => 0,
        'click'       => 0,
        'bounce'      => 0,
        'spam'        => 0,
        'unsubscribe' => 0,
        'sent_attempts' => 0,
        'target' => 0,
        'target_after_bounces' => 0,
    ];

    $stmt = $pdo->prepare("SELECT event_type, COUNT(*) as cnt FROM events WHERE campaign_id = ? GROUP BY event_type");
    $stmt->execute([$id]);
    foreach ($stmt as $row) {
        $etype = $row['event_type'];
        $cnt = (int)$row['cnt'];
        if ($etype === 'delivered') {
            $stats['delivered_raw'] = $cnt;
        } elseif ($etype === 'bounce') {
            $stats['bounce'] = $cnt;
        } elseif (isset($stats[$etype])) {
            $stats[$etype] = $cnt;
        }
    }

    $stats['delivered'] = max(0, $stats['delivered_raw'] - $stats['bounce']);
    $stats['sent_attempts'] = $stats['delivered_raw'] + $stats['bounce'];

    try {
        $stmt2 = $pdo->prepare("SELECT COUNT(*) FROM events WHERE campaign_id = ? AND event_type = ?");
        $stmt2->execute([$id, 'skipped_unsubscribe']);
        $skipped = (int)$stmt2->fetchColumn();
        if ($skipped > 0) {
            $stats['unsubscribe'] += $skipped;
        }
    } catch (Exception $e) {}

    try {
        $stmtc = $pdo->prepare("SELECT audience, total_recipients FROM campaigns WHERE id = ? LIMIT 1");
        $stmtc->execute([$id]);
        $camp = $stmtc->fetch(PDO::FETCH_ASSOC);
        $target = 0;
        if ($camp) {
            $aud = $camp['audience'] ?? '';
            if ($aud && strpos($aud, 'list:') === 0) {
                $lid = (int)substr($aud, strlen('list:'));
                if ($lid > 0) {
                    $stmtl = $pdo->prepare("SELECT COUNT(*) FROM contacts WHERE list_id = ?");
                    $stmtl->execute([$lid]);
                    $target = (int)$stmtl->fetchColumn();
                }
            } else {
                $target = (int)$camp['total_recipients'];
            }
        }
        $stats['target'] = $target;
        $stats['target_after_bounces'] = max(0, $target - $stats['bounce']);
    } catch (Exception $e) {}

    return $stats;
}

/**
 * Update campaign progress for real-time stats
 */
function update_campaign_progress(PDO $pdo, int $campaignId, int $sent, int $total, string $status = 'sending') {
    try {
        $stmt = $pdo->prepare("UPDATE campaigns SET progress_sent = ?, progress_total = ?, progress_status = ? WHERE id = ?");
        $stmt->execute([$sent, $total, $status, $campaignId]);
    } catch (Exception $e) {
        // Ignore errors gracefully (column might not exist in older schemas)
        error_log("Failed to update campaign progress: " . $e->getMessage());
    }
}

/**
 * Get campaign progress for real-time stats
 */
function get_campaign_progress(PDO $pdo, int $campaignId): array {
    try {
        $stmt = $pdo->prepare("SELECT progress_sent, progress_total, progress_status FROM campaigns WHERE id = ?");
        $stmt->execute([$campaignId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            return [
                'sent' => (int)$row['progress_sent'],
                'total' => (int)$row['progress_total'],
                'status' => $row['progress_status'] ?? 'draft',
                'percentage' => $row['progress_total'] > 0 ? round(($row['progress_sent'] / $row['progress_total']) * 100, 1) : 0
            ];
        }
    } catch (Exception $e) {
        // Graceful fallback for older schemas without progress columns
        error_log("Failed to get campaign progress: " . $e->getMessage());
    }
    return ['sent' => 0, 'total' => 0, 'status' => 'draft', 'percentage' => 0];
}

function get_profiles(PDO $pdo) {
    $stmt = $pdo->query("SELECT * FROM sending_profiles ORDER BY id ASC");
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function get_rotation_settings(PDO $pdo) {
    $stmt = $pdo->query("SELECT * FROM rotation_settings WHERE id = 1");
    $row  = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        $pdo->exec("INSERT INTO rotation_settings(id) VALUES(1)");
        $stmt = $pdo->query("SELECT * FROM rotation_settings WHERE id = 1");
        $row  = $stmt->fetch(PDO::FETCH_ASSOC);
    }
    return $row;
}

function update_rotation_settings(PDO $pdo, array $data) {
    $stmt = $pdo->prepare("
        UPDATE rotation_settings
        SET rotation_enabled = :rotation_enabled,
            mode = :mode,
            batch_size = :batch_size,
            max_sends_per_profile = :max_sends_per_profile,
            workers = :workers,
            messages_per_worker = :messages_per_worker
        WHERE id = 1
    ");
    $stmt->execute([
        ':rotation_enabled'      => $data['rotation_enabled'],
        ':mode'                  => $data['mode'],
        ':batch_size'            => $data['batch_size'],
        ':max_sends_per_profile' => $data['max_sends_per_profile'],
        ':workers'               => $data['workers'] ?? 4,
        ':messages_per_worker'   => $data['messages_per_worker'] ?? 100,
    ]);
}

function get_contact_lists(PDO $pdo) {
    $sql = "
        SELECT l.*, COUNT(c.id) AS contact_count
        FROM contact_lists l
        LEFT JOIN contacts c ON c.list_id = l.id
        GROUP BY l.id
        ORDER BY l.created_at DESC
    ";
    $stmt = $pdo->query($sql);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function get_contact_list(PDO $pdo, int $id) {
    $stmt = $pdo->prepare("SELECT * FROM contact_lists WHERE id = ?");
    $stmt->execute([$id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function get_contacts_for_list(PDO $pdo, int $listId) {
    $stmt = $pdo->prepare("SELECT * FROM contacts WHERE list_id = ? ORDER BY created_at DESC");
    $stmt->execute([$listId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function pick_next_profile(PDO $pdo) {
    $settings = get_rotation_settings($pdo);
    if ((int)$settings['rotation_enabled'] !== 1) {
        return null;
    }

    $profiles = get_profiles($pdo);
    $active = array_values(array_filter($profiles, function ($p) {
        return (int)$p['active'] === 1;
    }));

    if (empty($active)) return null;

    if ($settings['mode'] === 'random') {
        return $active[array_rand($active)];
    }

    $lastId = (int)$settings['last_profile_id'];
    $next = null;
    if ($lastId === 0) {
        $next = $active[0];
    } else {
        $foundIndex = null;
        foreach ($active as $i => $p) {
            if ((int)$p['id'] === $lastId) {
                $foundIndex = $i;
                break;
            }
        }
        if ($foundIndex === null || $foundIndex === (count($active)-1)) {
            $next = $active[0];
        } else {
            $next = $active[$foundIndex+1];
        }
    }

    if ($next) {
        $stmt = $pdo->prepare("UPDATE rotation_settings SET last_profile_id = ? WHERE id = 1");
        $stmt->execute([$next['id']]);
    }

    return $next;
}

function find_profile_for_campaign(PDO $pdo, array $campaign) {
    $from = trim($campaign['from_email'] ?? '');
    $profiles = get_profiles($pdo);

    if ($from !== '') {
        foreach ($profiles as $p) {
            if ((int)$p['active'] === 1 && strtolower($p['from_email']) === strtolower($from)) {
                return $p;
            }
        }
    }
    foreach ($profiles as $p) {
        if ((int)$p['active'] === 1) {
            return $p;
        }
    }
    return null;
}

/**
 * Check if an 'open' event exists for a campaign/recipient (safe PHP-based check)
 */
function has_open_event_for_rcpt(PDO $pdo, int $campaignId, string $rcpt): bool {
    if ($rcpt === '') return false;
    try {
        $stmt = $pdo->prepare("SELECT details FROM events WHERE campaign_id = ? AND event_type = 'open' LIMIT 2000");
        $stmt->execute([$campaignId]);
        foreach ($stmt as $row) {
            $d = json_decode($row['details'], true);
            if (is_array($d) && isset($d['rcpt']) && strtolower($d['rcpt']) === strtolower($rcpt)) {
                return true;
            }
        }
    } catch (Exception $e) {}
    return false;
}

/**
 * Buffered event logger for improved performance during high-volume sends
 * Batches event inserts to reduce database round-trips
 */
class BufferedEventLogger {
    private $pdo;
    private $buffer = [];
    private $bufferSize = 50; // Insert every 50 events
    private $campaignId;
    
    public function __construct(PDO $pdo, int $campaignId, int $bufferSize = 50) {
        $this->pdo = $pdo;
        $this->campaignId = $campaignId;
        $this->bufferSize = max(1, $bufferSize);
    }
    
    /**
     * Add an event to the buffer
     */
    public function log(string $eventType, array $details) {
        $this->buffer[] = [
            'campaign_id' => $this->campaignId,
            'event_type' => $eventType,
            'details' => json_encode($details)
        ];
        
        // Flush if buffer is full
        if (count($this->buffer) >= $this->bufferSize) {
            $this->flush();
        }
    }
    
    /**
     * Flush all buffered events to database
     */
    public function flush() {
        if (empty($this->buffer)) {
            return;
        }
        
        try {
            // Build batch insert
            $values = [];
            $params = [];
            foreach ($this->buffer as $event) {
                $values[] = "(?, ?, ?)";
                $params[] = $event['campaign_id'];
                $params[] = $event['event_type'];
                $params[] = $event['details'];
            }
            
            $sql = "INSERT INTO events (campaign_id, event_type, details) VALUES " . implode(", ", $values);
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            // Clear buffer
            $this->buffer = [];
        } catch (Exception $e) {
            error_log("BufferedEventLogger flush error: " . $e->getMessage());
            // Don't throw - we don't want to stop sending if logging fails
        }
    }
    
    /**
     * Destructor ensures buffer is flushed
     */
    public function __destruct() {
        $this->flush();
    }
}

/**
 * Main send function — unchanged semantics but tolerant to forced profile and structured results from senders.
 *
 * Added $profileOverrides parameter (associative array) that may contain per-profile runtime overrides:
 * e.g. ['send_rate' => <messages_per_second>]
 */
function send_campaign_real(PDO $pdo, array $campaign, string $recipientsText, bool $isTest = false, ?int $forceProfileId = null, array $profileOverrides = [])
{
    $recipients = array_filter(array_map('trim', preg_split("/\r\n|\n|\r|,/", $recipientsText)));
    if (empty($recipients)) {
        return;
    }

    $rotSettings      = get_rotation_settings($pdo);
    $rotationEnabled  = (int)$rotSettings['rotation_enabled'] === 1;

    $total = count($recipients);
    if (!$isTest) {
        try {
            $stmt = $pdo->prepare("UPDATE campaigns SET status='sending', total_recipients=? WHERE id=?");
            $stmt->execute([$total, $campaign['id']]);
        } catch (Exception $e) {}
    }

    $ins = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");

    $ok = 0;
    $failed = 0;

    $attempted = [];

    foreach ($recipients as $email) {
        $emailLower = strtolower(trim($email));
        if ($emailLower === '') continue;

        if (in_array($emailLower, $attempted, true)) {
            continue;
        }
        $attempted[] = $emailLower;

        try {
            if (is_unsubscribed($pdo, $emailLower)) {
                $ins->execute([
                    $campaign['id'],
                    'skipped_unsubscribe',
                    json_encode([
                        'rcpt' => $emailLower,
                        'reason' => 'recipient in unsubscribes table',
                        'test' => $isTest ? 1 : 0,
                    ])
                ]);
                $failed++;
                // increment sends_used? No - we didn't attempt to send.
                continue;
            }

            if ($forceProfileId !== null) {
                $stmtpf = $pdo->prepare("SELECT * FROM sending_profiles WHERE id = ? LIMIT 1");
                $stmtpf->execute([$forceProfileId]);
                $profile = $stmtpf->fetch(PDO::FETCH_ASSOC);
                if (!$profile) {
                    throw new Exception("Forced profile not found: {$forceProfileId}");
                }
                // Apply runtime overrides if provided (e.g., send_rate)
                if (!empty($profileOverrides) && isset($profileOverrides['send_rate'])) {
                    $profile['send_rate'] = (int)$profileOverrides['send_rate'];
                }
            } else {
                if ($rotationEnabled) {
                    $profile = pick_next_profile($pdo);
                } else {
                    $profile = find_profile_for_campaign($pdo, $campaign);
                }
            }

            if (!$profile) {
                throw new Exception("No active sending profile configured.");
            }

            // Refresh sends_used & max_sends from DB to ensure we don't exceed limit (protect concurrent runs)
            if (!empty($profile['id'])) {
                try {
                    $stmtSU = $pdo->prepare("SELECT COALESCE(sends_used,0) as su FROM sending_profiles WHERE id = ? LIMIT 1");
                    $stmtSU->execute([$profile['id']]);
                    $profile['sends_used'] = (int)$stmtSU->fetchColumn();
                } catch (Exception $e) {
                    $profile['sends_used'] = (int)($profile['sends_used'] ?? 0);
                }
            } else {
                $profile['sends_used'] = 0;
            }

            $maxSends = max(0, (int)($profile['max_sends'] ?? 0));
            if ($maxSends > 0 && $profile['sends_used'] >= $maxSends) {
                // profile exhausted - record skipped event and continue
                $ins->execute([
                    $campaign['id'],
                    'skipped_max_sends',
                    json_encode([
                        'rcpt' => $emailLower,
                        'profile_id' => $profile['id'] ?? null,
                        'reason' => 'profile reached max_sends',
                        'test' => $isTest ? 1 : 0,
                    ])
                ]);
                $failed++;
                continue;
            }

            // Resolve FROM address:
            // - If a profile is forced (worker per-profile) --> always use that profile's from_email.
            // - Else if rotation is enabled --> use the profile's from_email.
            // - Else (rotation disabled) --> prefer campaign's from_email, fallback to profile's from_email.
            $fromToUse = '';
            if ($forceProfileId !== null) {
                // Forced per-profile send: use profile's from
                $fromToUse = trim($profile['from_email'] ?? '');
            } elseif ($rotationEnabled) {
                // Global rotation enabled: each profile should send with its own From
                $fromToUse = trim($profile['from_email'] ?? '');
            } else {
                // Rotation disabled: keep the campaign-level From if provided, else fallback to profile
                $fromToUse = trim($campaign['from_email'] ?? '');
                if ($fromToUse === '') {
                    $fromToUse = trim($profile['from_email'] ?? '');
                }
            }

            if ($fromToUse === '') {
                throw new Exception("No FROM address resolved for this send.");
            }

            $campaignSend = $campaign;
            $campaignSend['from_email'] = $fromToUse;

            $campaignSend['sender_name'] = trim($profile['sender_name'] ?? '');
            if ($campaignSend['sender_name'] === '') {
                $campaignSend['sender_name'] = trim($campaign['sender_name'] ?? '');
            }

            $htmlTracked = build_tracked_html($campaignSend, $emailLower);

            // Respect overrides for send_rate (profileOverrides wins, then profile setting)
            $effectiveSendRate = 0;
            if (!empty($profileOverrides) && isset($profileOverrides['send_rate'])) {
                $effectiveSendRate = (int)$profileOverrides['send_rate'];
            } elseif (!empty($profile['send_rate'])) {
                $effectiveSendRate = (int)$profile['send_rate'];
            } else {
                $effectiveSendRate = 0;
            }

            if (isset($profile['type']) && $profile['type'] === 'api') {
                $res = api_send_mail($profile, $campaignSend, $emailLower, $htmlTracked);
            } else {
                $res = smtp_send_mail($profile, $campaignSend, $emailLower, $htmlTracked);
            }

            // After attempting send, increment sends_used (persist)
            if (!empty($profile['id'])) {
                try {
                    $stmtInc = $pdo->prepare("UPDATE sending_profiles SET sends_used = COALESCE(sends_used,0) + 1 WHERE id = ?");
                    $stmtInc->execute([$profile['id']]);
                } catch (Exception $ex) {
                    // ignore
                }
            }

            if (!is_array($res)) {
                $ins->execute([
                    $campaign['id'],
                    'bounce',
                    json_encode([
                        'rcpt'  => $emailLower,
                        'error' => 'Invalid send function response',
                        'profile_id' => isset($profile['id']) ? $profile['id'] : null,
                        'via'   => isset($profile['type']) ? $profile['type'] : null,
                        'test'  => $isTest ? 1 : 0,
                        'mode'  => 'sync',
                    ])
                ]);
                add_to_unsubscribes($pdo, $emailLower);
                try {
                    if (!empty($profile) && isset($profile['id']) && ($profile['type'] ?? '') === 'smtp') {
                        $stmt = $pdo->prepare("UPDATE sending_profiles SET active = 0 WHERE id = ?");
                        $stmt->execute([$profile['id']]);
                        $ins->execute([
                            $campaign['id'],
                            'profile_disabled',
                            json_encode([
                                'profile_id' => $profile['id'],
                                'reason' => 'invalid send function response',
                                'test' => $isTest ? 1 : 0,
                            ])
                        ]);
                    }
                } catch (Exception $ex) {}
                $failed++;
                continue;
            }

            if (!empty($res['ok']) && ($res['type'] ?? '') === 'delivered') {
                $ins->execute([
                    $campaign['id'],
                    'delivered',
                    json_encode([
                        'rcpt'       => $emailLower,
                        'profile_id' => $profile['id'] ?? null,
                        'via'        => $profile['type'] ?? null,
                        'smtp_code'  => $res['code'] ?? null,
                        'smtp_msg'   => $res['msg'] ?? '',
                        'stage'      => $res['stage'] ?? 'done',
                        'test'       => $isTest ? 1 : 0,
                        'mode'       => 'sync',
                    ])
                ]);
                $ok++;
            } else {
                $etype = ($res['type'] === 'deferred') ? 'deferred' : 'bounce';

                $ins->execute([
                    $campaign['id'],
                    $etype,
                    json_encode([
                        'rcpt'       => $emailLower,
                        'profile_id' => $profile['id'] ?? null,
                        'via'        => $profile['type'] ?? null,
                        'smtp_code'  => $res['code'] ?? null,
                        'smtp_msg'   => $res['msg'] ?? '',
                        'stage'      => $res['stage'] ?? 'unknown',
                        'test'       => $isTest ? 1 : 0,
                        'mode'       => 'sync',
                        'log'        => $res['log'] ?? [],
                    ])
                ]);

                if ($etype === 'bounce') {
                    add_to_unsubscribes($pdo, $emailLower);
                }

                try {
                    if (!empty($profile) && isset($profile['id']) && ($profile['type'] ?? '') === 'smtp' && $etype === 'bounce') {
                        $stmt = $pdo->prepare("UPDATE sending_profiles SET active = 0 WHERE id = ?");
                        $stmt->execute([$profile['id']]);

                        $ins->execute([
                            $campaign['id'],
                            'profile_disabled',
                            json_encode([
                                'profile_id' => $profile['id'],
                                'reason' => 'bounce/connection failure during send',
                                'test' => $isTest ? 1 : 0,
                            ])
                        ]);
                    }
                } catch (Exception $ex) { /* ignore */ }

                $failed++;
            }
        } catch (Exception $e) {
            $ins->execute([
                $campaign['id'],
                'bounce',
                json_encode([
                    'rcpt'        => $emailLower,
                    'error'       => $e->getMessage(),
                    'profile_id'  => isset($profile['id']) ? $profile['id'] : null,
                    'via'         => isset($profile['type']) ? $profile['type'] : null,
                    'test'        => $isTest ? 1 : 0,
                    'mode'        => 'exception',
                ])
            ]);
            add_to_unsubscribes($pdo, $emailLower);

            try {
                if (!empty($profile) && isset($profile['id']) && ($profile['type'] ?? '') === 'smtp') {
                    $stmt = $pdo->prepare("UPDATE sending_profiles SET active = 0 WHERE id = ?");
                    $stmt->execute([$profile['id']]);
                }
            } catch (Exception $ex) {}

            // increment sends_used even on exception to count attempt
            if (!empty($profile['id'])) {
                try {
                    $stmtInc = $pdo->prepare("UPDATE sending_profiles SET sends_used = COALESCE(sends_used,0) + 1 WHERE id = ?");
                    $stmtInc->execute([$profile['id']]);
                } catch (Exception $_) {}
            }

            $failed++;
        }

        // Throttle according to effective send rate (overrides or profile)
        $sendRate = 0;
        if (!empty($profileOverrides) && isset($profileOverrides['send_rate'])) {
            $sendRate = (int)$profileOverrides['send_rate'];
        } elseif (!empty($profile) && isset($profile['send_rate'])) {
            $sendRate = (int)$profile['send_rate'];
        } else {
            $sendRate = 0;
        }

        if ($sendRate > 0) {
            $micro = (int)(1000000 / max(1, $sendRate));
            if ($micro > 0) {
                usleep($micro);
            }
        }
    }

    // Final status update - CRITICAL: This must always execute
    if (!$isTest) {
        try {
            $stmt = $pdo->prepare("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id=?");
            $result = $stmt->execute([$campaign['id']]);
            if (!$result) {
                error_log("Failed to update campaign status to sent for campaign ID: {$campaign['id']}");
                // Force update with direct query as fallback
                $pdo->exec("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id={$campaign['id']}");
            } else {
                error_log("Successfully updated campaign {$campaign['id']} status to sent (simple mode)");
            }
        } catch (Exception $e) {
            error_log("Exception updating campaign status to sent (simple): " . $e->getMessage());
            // Force update with direct query as last resort
            try {
                $pdo->exec("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id={$campaign['id']}");
                error_log("Fallback status update successful for campaign {$campaign['id']}");
            } catch (Exception $e2) {
                error_log("Fallback status update also failed: " . $e2->getMessage());
            }
        }
    }
}

///////////////////////
//  CONCURRENT EMAIL SENDING ARCHITECTURE
///////////////////////
/**
 * CONCURRENT EMAIL SENDING SYSTEM
 * ================================
 * 
 * This system implements high-performance concurrent email sending using PHP's pcntl extension
 * for process forking. When pcntl is not available, it falls back to sequential processing.
 * 
 * ARCHITECTURE OVERVIEW:
 * 
 * 1. MULTI-LEVEL PARALLELIZATION
 *    - Level 1: Profile-level parallelization (rotation mode)
 *      When rotation is enabled with multiple profiles, each profile gets a separate process
 *    - Level 2: Connection-level parallelization (within each profile)
 *      Each profile spawns N workers based on connection_numbers setting
 *    - Level 3: Batch-level persistent connections (within each worker)
 *      Each worker sends batch_size emails before reconnecting to SMTP
 * 
 * 2. PROCESS HIERARCHY
 *    Main Process
 *    ├── Profile Worker 1 (if rotation enabled)
 *    │   ├── Connection Worker 1
 *    │   ├── Connection Worker 2
 *    │   └── Connection Worker N
 *    ├── Profile Worker 2
 *    │   └── ...
 *    └── Profile Worker M
 * 
 * 3. KEY FUNCTIONS
 *    - send_campaign_with_rotation(): Entry point for multi-profile campaigns
 *    - send_campaign_with_connections(): Entry point for single-profile campaigns
 *    - send_campaign_with_connections_for_profile_concurrent(): Spawns connection workers
 *    - process_worker_batch(): Worker function that runs in forked process
 * 
 * 4. CONFIGURATION
 *    - connection_numbers: Number of concurrent SMTP connections (1-40, default 5)
 *    - batch_size: Emails per connection before reconnecting (1-500, default 50)
 *    - rotation_enabled: Use multiple profiles concurrently
 * 
 * 5. THREAD SAFETY
 *    - Each worker creates its own database connection
 *    - Workers log events independently to avoid contention
 *    - Parent processes count results via database queries
 *    - Worker PIDs tracked in event details for debugging
 * 
 * 6. PERFORMANCE CHARACTERISTICS
 *    Example: 10,000 emails, 2 profiles, 10 connections each, batch size 50
 *    - 2 profile processes (parallel)
 *    - Each spawns 10 connection workers (parallel within profile)
 *    - Total: 20 concurrent workers
 *    - Each sends 50 emails before reconnecting
 *    - Theoretical: ~20x speedup vs sequential
 * 
 * 7. FALLBACK BEHAVIOR
 *    - If pcntl_fork unavailable: Sequential processing with same batch optimization
 *    - If fork fails: Logs error and continues with remaining workers
 *    - Test mode: Always sequential for predictability
 */

/**
 * Worker function to process a batch of emails in a separate process
 * This runs in a forked child process and exits when done
 * 
 * @param array $dbConfig Database configuration
 * @param array $profile SMTP profile
 * @param array $campaign Campaign data
 * @param array $recipients Recipients for this worker
 * @param int $batchSize Batch size for reconnecting
 * @param bool $isTest Whether this is a test send
 */
function process_worker_batch(array $dbConfig, array $profile, array $campaign, array $recipients, int $batchSize, bool $isTest = false) {
    // Create new PDO connection for this worker process
    try {
        $pdo = new PDO(
            "mysql:host={$dbConfig['host']};dbname={$dbConfig['name']};charset=utf8mb4",
            $dbConfig['user'],
            $dbConfig['pass'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
    } catch (Exception $e) {
        error_log("Worker PDO error: " . $e->getMessage());
        exit(1);
    }
    
    $campaignId = (int)$campaign['id'];
    $ins = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");
    
    $sent = 0;
    $failed = 0;
    
    // Prepare campaign for sending
    $campaignSend = $campaign;
    $campaignSend['from_email'] = trim($profile['from_email'] ?? $campaign['from_email']);
    $campaignSend['sender_name'] = trim($profile['sender_name'] ?? $campaign['sender_name']);
    
    // Split into batches based on batch_size for reconnecting
    $smtpBatches = array_chunk($recipients, $batchSize);
    
    foreach ($smtpBatches as $batch) {
        // Build HTML map for each recipient in the batch
        $htmlMap = [];
        foreach ($batch as $emailLower) {
            $htmlMap[$emailLower] = build_tracked_html($campaignSend, $emailLower);
        }
        
        try {
            if (isset($profile['type']) && $profile['type'] === 'api') {
                // API sending - send individually
                foreach ($batch as $emailLower) {
                    $res = api_send_mail($profile, $campaignSend, $emailLower, $htmlMap[$emailLower]);
                    
                    if (!empty($res['ok']) && ($res['type'] ?? '') === 'delivered') {
                        $ins->execute([
                            $campaignId,
                            'delivered',
                            json_encode([
                                'rcpt' => $emailLower,
                                'profile_id' => $profile['id'] ?? null,
                                'via' => 'concurrent_' . ($profile['type'] ?? 'api'),
                                'smtp_code' => $res['code'] ?? null,
                                'test' => $isTest ? 1 : 0,
                                'worker_pid' => getmypid(),
                            ])
                        ]);
                        $sent++;
                    } else {
                        $etype = ($res['type'] === 'deferred') ? 'deferred' : 'bounce';
                        $ins->execute([
                            $campaignId,
                            $etype,
                            json_encode([
                                'rcpt' => $emailLower,
                                'profile_id' => $profile['id'] ?? null,
                                'via' => 'concurrent_' . ($profile['type'] ?? 'api'),
                                'smtp_code' => $res['code'] ?? null,
                                'smtp_msg' => $res['msg'] ?? '',
                                'test' => $isTest ? 1 : 0,
                                'worker_pid' => getmypid(),
                            ])
                        ]);
                        if ($etype === 'bounce') {
                            add_to_unsubscribes($pdo, $emailLower);
                        }
                        $failed++;
                    }
                }
            } else {
                // SMTP batch sending with persistent connection
                $batchResult = smtp_send_batch($profile, $campaignSend, $batch, $htmlMap);
                
                if (isset($batchResult['results']) && is_array($batchResult['results'])) {
                    foreach ($batchResult['results'] as $recipientEmail => $res) {
                        if (!empty($res['ok']) && ($res['type'] ?? '') === 'delivered') {
                            $ins->execute([
                                $campaignId,
                                'delivered',
                                json_encode([
                                    'rcpt' => $recipientEmail,
                                    'profile_id' => $profile['id'] ?? null,
                                    'via' => 'concurrent_smtp_batch',
                                    'smtp_code' => $res['code'] ?? null,
                                    'test' => $isTest ? 1 : 0,
                                    'worker_pid' => getmypid(),
                                ])
                            ]);
                            $sent++;
                        } else {
                            $etype = ($res['type'] === 'deferred') ? 'deferred' : 'bounce';
                            $ins->execute([
                                $campaignId,
                                $etype,
                                json_encode([
                                    'rcpt' => $recipientEmail,
                                    'profile_id' => $profile['id'] ?? null,
                                    'via' => 'concurrent_smtp_batch',
                                    'smtp_code' => $res['code'] ?? null,
                                    'smtp_msg' => $res['msg'] ?? '',
                                    'test' => $isTest ? 1 : 0,
                                    'worker_pid' => getmypid(),
                                ])
                            ]);
                            if ($etype === 'bounce') {
                                add_to_unsubscribes($pdo, $recipientEmail);
                            }
                            $failed++;
                        }
                    }
                } else {
                    // Batch failed entirely
                    foreach ($batch as $emailLower) {
                        $ins->execute([
                            $campaignId,
                            'bounce',
                            json_encode([
                                'rcpt' => $emailLower,
                                'error' => $batchResult['error'] ?? 'Batch connection failed',
                                'profile_id' => $profile['id'] ?? null,
                                'via' => 'concurrent_smtp_batch',
                                'test' => $isTest ? 1 : 0,
                                'worker_pid' => getmypid(),
                            ])
                        ]);
                        add_to_unsubscribes($pdo, $emailLower);
                        $failed++;
                    }
                }
            }
        } catch (Exception $e) {
            foreach ($batch as $emailLower) {
                $ins->execute([
                    $campaignId,
                    'bounce',
                    json_encode([
                        'rcpt' => $emailLower,
                        'error' => 'Worker exception: ' . $e->getMessage(),
                        'profile_id' => $profile['id'] ?? null,
                        'test' => $isTest ? 1 : 0,
                        'worker_pid' => getmypid(),
                    ])
                ]);
                add_to_unsubscribes($pdo, $emailLower);
                $failed++;
            }
        }
    }
    
    exit(0);
}

/**
 * Concurrent version of send_campaign_with_connections_for_profile
 * Uses process forking to send via multiple concurrent connections
 */
function send_campaign_with_connections_for_profile_concurrent(PDO $pdo, array $campaign, array $recipients, array $profile, bool $isTest = false) {
    global $DB_HOST, $DB_NAME, $DB_USER, $DB_PASS;
    
    if (empty($recipients)) {
        return ['sent' => 0, 'failed' => 0];
    }
    
    $campaignId = (int)$campaign['id'];
    
    // Get connection numbers and batch size
    $connectionNumbers = max(MIN_CONNECTIONS, min(MAX_CONNECTIONS, (int)($profile['connection_numbers'] ?? DEFAULT_CONNECTIONS)));
    $batchSize = max(MIN_BATCH_SIZE, min(MAX_BATCH_SIZE, (int)($profile['batch_size'] ?? DEFAULT_BATCH_SIZE)));
    
    $ins = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");
    
    // Filter out unsubscribed recipients
    $validRecipients = [];
    $failed = 0;
    foreach ($recipients as $email) {
        $emailLower = strtolower(trim($email));
        if ($emailLower === '' || in_array($emailLower, $validRecipients, true)) {
            continue;
        }
        
        if (is_unsubscribed($pdo, $emailLower)) {
            $ins->execute([
                $campaignId,
                'skipped_unsubscribe',
                json_encode([
                    'rcpt' => $emailLower,
                    'reason' => 'unsubscribed',
                    'test' => $isTest ? 1 : 0,
                ])
            ]);
            $failed++;
            continue;
        }
        
        $validRecipients[] = $emailLower;
    }
    
    // Divide recipients across connections
    $totalValidRecipients = count($validRecipients);
    if ($totalValidRecipients === 0) {
        return ['sent' => 0, 'failed' => $failed];
    }
    
    $recipientsPerConnection = ceil($totalValidRecipients / $connectionNumbers);
    
    // Check if pcntl extension is available for true concurrency
    $canFork = function_exists('pcntl_fork') && function_exists('pcntl_waitpid');
    
    if (!$canFork) {
        // Fallback to sequential processing if pcntl not available
        return send_campaign_with_connections_for_profile($pdo, $campaign, $recipients, $profile, $isTest);
    }
    
    // Prepare database config for workers
    $dbConfig = [
        'host' => $DB_HOST,
        'name' => $DB_NAME,
        'user' => $DB_USER,
        'pass' => $DB_PASS,
    ];
    
    $workerPids = [];
    
    // Spawn worker processes for each connection
    for ($i = 0; $i < $connectionNumbers; $i++) {
        $start = $i * $recipientsPerConnection;
        if ($start >= $totalValidRecipients) break;
        
        $connectionRecipients = array_slice($validRecipients, $start, $recipientsPerConnection);
        if (empty($connectionRecipients)) continue;
        
        $pid = pcntl_fork();
        
        if ($pid === -1) {
            // Fork failed - log error and continue with next
            error_log("Failed to fork worker process for connection $i");
            continue;
        } elseif ($pid === 0) {
            // Child process - process this batch
            process_worker_batch($dbConfig, $profile, $campaign, $connectionRecipients, $batchSize, $isTest);
            // process_worker_batch calls exit(), so we never reach here
        } else {
            // Parent process - store child PID
            $workerPids[] = $pid;
        }
    }
    
    // Parent process: wait for all workers to complete
    $sent = 0;
    foreach ($workerPids as $pid) {
        $status = 0;
        pcntl_waitpid($pid, $status);
        // Note: We can't easily get sent/failed counts from child processes
        // The database events table will have the accurate data
    }
    
    // Count results from database for accurate totals
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) 
            FROM events 
            WHERE campaign_id = ? 
            AND event_type = 'delivered' 
            AND JSON_EXTRACT(details, '$.worker_pid') IS NOT NULL
            AND created_at >= DATE_SUB(NOW(), INTERVAL 10 MINUTE)
        ");
        $stmt->execute([$campaignId]);
        $sent = (int)$stmt->fetchColumn();
    } catch (Exception $e) {
        $sent = 0;
    }
    
    return ['sent' => $sent, 'failed' => $failed];
}

/**
 * Multi-profile sending function for rotation mode
 * 
 * When rotation is enabled, this function distributes recipients evenly across ALL active profiles.
 * Each profile then uses its own connection_numbers to further parallelize sending.
 * 
 * Example: 1000 emails, 4 active profiles → 250 emails per profile
 * If profile 1 has 10 connections → 25 emails per connection within profile 1
 * 
 * Status flow: draft → queued (orange) → sending (yellow) → sent (green)
 */
function send_campaign_with_rotation(PDO $pdo, array $campaign, array $recipients, array $activeProfiles, bool $isTest = false) {
    if (empty($recipients) || empty($activeProfiles)) {
        return ['sent' => 0, 'failed' => count($recipients)];
    }
    
    $total = count($recipients);
    $campaignId = (int)$campaign['id'];
    
    // PHASE 1: QUEUED - Set status to queued and prepare batches across profiles
    if (!$isTest) {
        update_campaign_progress($pdo, $campaignId, 0, $total, 'queued');
        try {
            $stmt = $pdo->prepare("UPDATE campaigns SET status='queued', total_recipients=? WHERE id=?");
            $stmt->execute([$total, $campaignId]);
        } catch (Exception $e) {}
    }
    
    // Divide recipients evenly across all active profiles
    $profileCount = count($activeProfiles);
    $recipientsPerProfile = ceil($total / $profileCount);
    
    $profileBatches = [];
    for ($i = 0; $i < $profileCount; $i++) {
        $start = $i * $recipientsPerProfile;
        if ($start >= $total) break;
        
        $profileRecipients = array_slice($recipients, $start, $recipientsPerProfile);
        if (!empty($profileRecipients)) {
            $profileBatches[] = [
                'profile' => $activeProfiles[$i],
                'recipients' => $profileRecipients
            ];
        }
    }
    
    // PHASE 2: SENDING - Update status to sending
    if (!$isTest) {
        update_campaign_progress($pdo, $campaignId, 0, $total, 'sending');
        try {
            $stmt = $pdo->prepare("UPDATE campaigns SET status='sending' WHERE id=?");
            $stmt->execute([$campaignId]);
        } catch (Exception $e) {}
    }
    
    // Check if we can fork processes for concurrent profile sending
    $canFork = function_exists('pcntl_fork') && function_exists('pcntl_waitpid');
    
    $totalSent = 0;
    $totalFailed = 0;
    
    if ($canFork && !$isTest) {
        // CONCURRENT MODE: Fork a process for each profile
        global $DB_HOST, $DB_NAME, $DB_USER, $DB_PASS;
        
        $profilePids = [];
        
        foreach ($profileBatches as $batch) {
            $profile = $batch['profile'];
            $profileRecipients = $batch['recipients'];
            
            $pid = pcntl_fork();
            
            if ($pid === -1) {
                // Fork failed - fall back to sequential for this profile
                error_log("Failed to fork process for profile {$profile['id']}");
                $result = send_campaign_with_connections_for_profile_concurrent(
                    $pdo, 
                    $campaign, 
                    $profileRecipients, 
                    $profile, 
                    $isTest
                );
                $totalSent += $result['sent'];
                $totalFailed += $result['failed'];
            } elseif ($pid === 0) {
                // Child process - handle this profile
                // Create new PDO connection for child
                try {
                    $childPdo = new PDO(
                        "mysql:host={$DB_HOST};dbname={$DB_NAME};charset=utf8mb4",
                        $DB_USER,
                        $DB_PASS,
                        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                    );
                } catch (Exception $e) {
                    error_log("Child PDO error: " . $e->getMessage());
                    exit(1);
                }
                
                // Process this profile's recipients with concurrent connections
                send_campaign_with_connections_for_profile_concurrent(
                    $childPdo, 
                    $campaign, 
                    $profileRecipients, 
                    $profile, 
                    $isTest
                );
                
                exit(0);
            } else {
                // Parent process - store child PID
                $profilePids[] = $pid;
            }
        }
        
        // Wait for all profile workers to complete
        foreach ($profilePids as $pid) {
            $status = 0;
            pcntl_waitpid($pid, $status);
        }
        
        // Count results from database
        try {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE campaign_id = ? AND event_type = 'delivered'");
            $stmt->execute([$campaignId]);
            $totalSent = (int)$stmt->fetchColumn();
            
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE campaign_id = ? AND event_type IN ('bounce', 'deferred', 'skipped_unsubscribe')");
            $stmt->execute([$campaignId]);
            $totalFailed = (int)$stmt->fetchColumn();
        } catch (Exception $e) {
            error_log("Error counting results: " . $e->getMessage());
        }
    } else {
        // SEQUENTIAL MODE: Process each profile one by one (fallback or test mode)
        foreach ($profileBatches as $batch) {
            $profile = $batch['profile'];
            $profileRecipients = $batch['recipients'];
            
            // Use concurrent connections within each profile
            $result = send_campaign_with_connections_for_profile_concurrent(
                $pdo, 
                $campaign, 
                $profileRecipients, 
                $profile, 
                $isTest
            );
            
            $totalSent += $result['sent'];
            $totalFailed += $result['failed'];
        }
    }
    
    // Final status update - CRITICAL: This must always execute
    // Wrapped in try-finally to ensure execution even if errors occur
    try {
        if (!$isTest) {
            // Mark as completed in progress tracker
            update_campaign_progress($pdo, $campaignId, $totalSent + $totalFailed, $total, 'completed');
            
            // Update main campaign status to 'sent'
            try {
                $stmt = $pdo->prepare("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id=?");
                $result = $stmt->execute([$campaignId]);
                if (!$result) {
                    error_log("Failed to update campaign status to sent for campaign ID: {$campaignId}");
                    // Force update with direct query as fallback
                    $pdo->exec("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id={$campaignId}");
                } else {
                    error_log("Successfully updated campaign {$campaignId} status to sent (rotation mode)");
                }
            } catch (Exception $e) {
                error_log("Exception updating campaign status to sent (rotation): " . $e->getMessage());
                // Force update with direct query as last resort
                try {
                    $pdo->exec("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id={$campaignId}");
                    error_log("Fallback status update successful for campaign {$campaignId}");
                } catch (Exception $e2) {
                    error_log("Fallback status update also failed: " . $e2->getMessage());
                }
            }
        }
    } finally {
        // Ensure we always log completion regardless of any errors
        error_log("Campaign {$campaignId} rotation send completed: sent={$totalSent}, failed={$totalFailed}, total={$total}");
    }
    
    return ['sent' => $totalSent, 'failed' => $totalFailed];
}

/**
 * Helper function for sending with a specific profile (used by multi-profile rotation)
 * Does NOT update campaign status - that's handled by the caller
 */
function send_campaign_with_connections_for_profile(PDO $pdo, array $campaign, array $recipients, array $profile, bool $isTest = false) {
    if (empty($recipients)) {
        return ['sent' => 0, 'failed' => 0];
    }
    
    $campaignId = (int)$campaign['id'];
    
    // Get connection numbers (MaxBulk Mailer style - number of concurrent connections)
    $connectionNumbers = max(MIN_CONNECTIONS, min(MAX_CONNECTIONS, (int)($profile['connection_numbers'] ?? DEFAULT_CONNECTIONS)));
    
    // Get batch size (number of emails to send per single SMTP connection before reconnecting)
    $batchSize = max(MIN_BATCH_SIZE, min(MAX_BATCH_SIZE, (int)($profile['batch_size'] ?? DEFAULT_BATCH_SIZE)));
    
    $ins = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");
    
    $sent = 0;
    $failed = 0;
    $attempted = [];
    
    // Filter out unsubscribed and prepare recipients
    $validRecipients = [];
    foreach ($recipients as $email) {
        $emailLower = strtolower(trim($email));
        if ($emailLower === '' || in_array($emailLower, $attempted, true)) {
            continue;
        }
        $attempted[] = $emailLower;
        
        // Check unsubscribed
        if (is_unsubscribed($pdo, $emailLower)) {
            $ins->execute([
                $campaignId,
                'skipped_unsubscribe',
                json_encode([
                    'rcpt' => $emailLower,
                    'reason' => 'unsubscribed',
                    'test' => $isTest ? 1 : 0,
                ])
            ]);
            $failed++;
            continue;
        }
        
        $validRecipients[] = $emailLower;
    }
    
    // BATCHING MECHANISM: Divide recipients across connections for this profile
    $totalValidRecipients = count($validRecipients);
    $recipientsPerConnection = ceil($totalValidRecipients / $connectionNumbers);
    
    // Split recipients into connection-based batches
    $connectionBatches = [];
    for ($i = 0; $i < $connectionNumbers; $i++) {
        $start = $i * $recipientsPerConnection;
        if ($start >= $totalValidRecipients) break;
        
        $connectionRecipients = array_slice($validRecipients, $start, $recipientsPerConnection);
        if (!empty($connectionRecipients)) {
            $connectionBatches[] = $connectionRecipients;
        }
    }
    
    // Prepare campaign for sending
    $campaignSend = $campaign;
    $campaignSend['from_email'] = trim($profile['from_email'] ?? $campaign['from_email']);
    $campaignSend['sender_name'] = trim($profile['sender_name'] ?? $campaign['sender_name']);
    
    // Process each connection batch
    foreach ($connectionBatches as $connectionIdx => $connectionRecipients) {
        // Within each connection, split into smaller batches based on batch_size
        $smtpBatches = array_chunk($connectionRecipients, $batchSize);
        
        foreach ($smtpBatches as $batch) {
            // Build HTML map for each recipient in the batch
            $htmlMap = [];
            foreach ($batch as $emailLower) {
                $htmlMap[$emailLower] = build_tracked_html($campaignSend, $emailLower);
            }
            
            try {
                // Send batch using single connection
                if (isset($profile['type']) && $profile['type'] === 'api') {
                    // API sending doesn't benefit from persistent connections, send individually
                    foreach ($batch as $emailLower) {
                        $res = api_send_mail($profile, $campaignSend, $emailLower, $htmlMap[$emailLower]);
                        
                        if (!empty($res['ok']) && ($res['type'] ?? '') === 'delivered') {
                            $ins->execute([
                                $campaignId,
                                'delivered',
                                json_encode([
                                    'rcpt' => $emailLower,
                                    'profile_id' => $profile['id'] ?? null,
                                    'via' => $profile['type'] ?? null,
                                    'smtp_code' => $res['code'] ?? null,
                                    'test' => $isTest ? 1 : 0,
                                ])
                            ]);
                            $sent++;
                        } else {
                            $etype = ($res['type'] === 'deferred') ? 'deferred' : 'bounce';
                            $ins->execute([
                                $campaignId,
                                $etype,
                                json_encode([
                                    'rcpt' => $emailLower,
                                    'profile_id' => $profile['id'] ?? null,
                                    'via' => $profile['type'] ?? null,
                                    'smtp_code' => $res['code'] ?? null,
                                    'smtp_msg' => $res['msg'] ?? '',
                                    'test' => $isTest ? 1 : 0,
                                ])
                            ]);
                            
                            if ($etype === 'bounce') {
                                add_to_unsubscribes($pdo, $emailLower);
                            }
                            $failed++;
                        }
                    }
                } else {
                    // SMTP batch sending with persistent connection
                    $batchResult = smtp_send_batch($profile, $campaignSend, $batch, $htmlMap);
                    
                    if (isset($batchResult['results']) && is_array($batchResult['results'])) {
                        foreach ($batchResult['results'] as $recipientEmail => $res) {
                            if (!empty($res['ok']) && ($res['type'] ?? '') === 'delivered') {
                                $ins->execute([
                                    $campaignId,
                                    'delivered',
                                    json_encode([
                                        'rcpt' => $recipientEmail,
                                        'profile_id' => $profile['id'] ?? null,
                                        'via' => 'smtp_batch',
                                        'smtp_code' => $res['code'] ?? null,
                                        'test' => $isTest ? 1 : 0,
                                    ])
                                ]);
                                $sent++;
                            } else {
                                $etype = ($res['type'] === 'deferred') ? 'deferred' : 'bounce';
                                $ins->execute([
                                    $campaignId,
                                    $etype,
                                    json_encode([
                                        'rcpt' => $recipientEmail,
                                        'profile_id' => $profile['id'] ?? null,
                                        'via' => 'smtp_batch',
                                        'smtp_code' => $res['code'] ?? null,
                                        'smtp_msg' => $res['msg'] ?? '',
                                        'test' => $isTest ? 1 : 0,
                                    ])
                                ]);
                                
                                if ($etype === 'bounce') {
                                    add_to_unsubscribes($pdo, $recipientEmail);
                                }
                                $failed++;
                            }
                        }
                    } else {
                        // Batch failed entirely (connection error)
                        foreach ($batch as $emailLower) {
                            $ins->execute([
                                $campaignId,
                                'bounce',
                                json_encode([
                                    'rcpt' => $emailLower,
                                    'error' => $batchResult['error'] ?? 'Batch connection failed',
                                    'profile_id' => $profile['id'] ?? null,
                                    'via' => 'smtp_batch',
                                    'test' => $isTest ? 1 : 0,
                                ])
                            ]);
                            add_to_unsubscribes($pdo, $emailLower);
                            $failed++;
                        }
                    }
                }
            } catch (Exception $e) {
                // Handle batch exception
                foreach ($batch as $emailLower) {
                    $ins->execute([
                        $campaignId,
                        'bounce',
                        json_encode([
                            'rcpt' => $emailLower,
                            'error' => 'Batch exception: ' . $e->getMessage(),
                            'profile_id' => $profile['id'] ?? null,
                            'test' => $isTest ? 1 : 0,
                        ])
                    ]);
                    add_to_unsubscribes($pdo, $emailLower);
                    $failed++;
                }
            }
        }
    }
    
    return ['sent' => $sent, 'failed' => $failed];
}

/**
 * Optimized sending function using connection-based batching with persistent connections
 * 
 * Implements a two-phase process:
 * 1. QUEUED phase: Divides recipients evenly across connection_numbers (e.g., 40 connections = 40 batches)
 * 2. SENDING phase: Sends emails using persistent SMTP connections, reconnecting every batch_size emails
 * 
 * Status flow: draft → queued (orange) → sending (yellow) → sent (green)
 */
function send_campaign_with_connections(PDO $pdo, array $campaign, array $recipients, ?int $profileId = null, bool $isTest = false) {
    if (empty($recipients)) {
        return ['sent' => 0, 'failed' => 0];
    }
    
    $total = count($recipients);
    $campaignId = (int)$campaign['id'];
    
    // Get profile first to determine connection numbers
    $profile = null;
    if ($profileId) {
        $stmt = $pdo->prepare("SELECT * FROM sending_profiles WHERE id = ? LIMIT 1");
        $stmt->execute([$profileId]);
        $profile = $stmt->fetch(PDO::FETCH_ASSOC);
    } else {
        $profile = find_profile_for_campaign($pdo, $campaign);
    }
    
    if (!$profile) {
        return ['sent' => 0, 'failed' => $total, 'error' => 'No active profile'];
    }
    
    // PHASE 1: QUEUED - Set status to queued and prepare batches
    if (!$isTest) {
        update_campaign_progress($pdo, $campaignId, 0, $total, 'queued');
        try {
            $stmt = $pdo->prepare("UPDATE campaigns SET status='queued', total_recipients=? WHERE id=?");
            $stmt->execute([$total, $campaignId]);
        } catch (Exception $e) {}
    }
    
    // Get connection numbers (MaxBulk Mailer style - number of concurrent connections)
    $connectionNumbers = max(MIN_CONNECTIONS, min(MAX_CONNECTIONS, (int)($profile['connection_numbers'] ?? DEFAULT_CONNECTIONS)));
    
    // Get batch size (number of emails to send per single SMTP connection before reconnecting)
    $batchSize = max(MIN_BATCH_SIZE, min(MAX_BATCH_SIZE, (int)($profile['batch_size'] ?? DEFAULT_BATCH_SIZE)));
    
    $ins = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");
    
    $sent = 0;
    $failed = 0;
    $attempted = [];
    
    // Filter out unsubscribed and prepare recipients
    $validRecipients = [];
    foreach ($recipients as $email) {
        $emailLower = strtolower(trim($email));
        if ($emailLower === '' || in_array($emailLower, $attempted, true)) {
            continue;
        }
        $attempted[] = $emailLower;
        
        // Check unsubscribed
        if (is_unsubscribed($pdo, $emailLower)) {
            $ins->execute([
                $campaignId,
                'skipped_unsubscribe',
                json_encode([
                    'rcpt' => $emailLower,
                    'reason' => 'unsubscribed',
                    'test' => $isTest ? 1 : 0,
                ])
            ]);
            $failed++;
            continue;
        }
        
        $validRecipients[] = $emailLower;
    }
    
    // BATCHING MECHANISM: Divide recipients across connections
    // Each connection gets approximately equal number of emails
    $totalValidRecipients = count($validRecipients);
    $recipientsPerConnection = ceil($totalValidRecipients / $connectionNumbers);
    
    // Split recipients into connection-based batches
    $connectionBatches = [];
    for ($i = 0; $i < $connectionNumbers; $i++) {
        $start = $i * $recipientsPerConnection;
        if ($start >= $totalValidRecipients) break;
        
        $connectionRecipients = array_slice($validRecipients, $start, $recipientsPerConnection);
        if (!empty($connectionRecipients)) {
            $connectionBatches[] = $connectionRecipients;
        }
    }
    
    // PHASE 2: SENDING - Update status to sending when we start actual sending
    if (!$isTest) {
        update_campaign_progress($pdo, $campaignId, 0, $total, 'sending');
        try {
            $stmt = $pdo->prepare("UPDATE campaigns SET status='sending' WHERE id=?");
            $stmt->execute([$campaignId]);
        } catch (Exception $e) {}
    }
    
    // Prepare campaign for sending
    $campaignSend = $campaign;
    $campaignSend['from_email'] = trim($profile['from_email'] ?? $campaign['from_email']);
    $campaignSend['sender_name'] = trim($profile['sender_name'] ?? $campaign['sender_name']);
    
    // Check if we can use concurrent processing
    $canFork = function_exists('pcntl_fork') && function_exists('pcntl_waitpid');
    
    if ($canFork && !$isTest && count($connectionBatches) > 1) {
        // CONCURRENT MODE: Fork a worker for each connection batch
        global $DB_HOST, $DB_NAME, $DB_USER, $DB_PASS;
        
        $dbConfig = [
            'host' => $DB_HOST,
            'name' => $DB_NAME,
            'user' => $DB_USER,
            'pass' => $DB_PASS,
        ];
        
        $workerPids = [];
        
        foreach ($connectionBatches as $connectionIdx => $connectionRecipients) {
            $pid = pcntl_fork();
            
            if ($pid === -1) {
                // Fork failed - log and continue with sequential fallback
                error_log("Failed to fork worker for connection $connectionIdx");
                continue;
            } elseif ($pid === 0) {
                // Child process - send this batch
                process_worker_batch($dbConfig, $profile, $campaignSend, $connectionRecipients, $batchSize, $isTest);
                // process_worker_batch calls exit()
            } else {
                // Parent process
                $workerPids[] = $pid;
            }
        }
        
        // Wait for all workers
        foreach ($workerPids as $pid) {
            $status = 0;
            pcntl_waitpid($pid, $status);
        }
        
        // Count results from database (both sent and failed)
        try {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE campaign_id = ? AND event_type = 'delivered'");
            $stmt->execute([$campaignId]);
            $sent = (int)$stmt->fetchColumn();
            
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE campaign_id = ? AND event_type IN ('bounce', 'deferred', 'skipped_unsubscribe')");
            $stmt->execute([$campaignId]);
            $failed = (int)$stmt->fetchColumn();
        } catch (Exception $e) {
            $sent = 0;
            $failed = 0;
        }
    } else {
        // SEQUENTIAL MODE: Process each connection batch sequentially (fallback or test mode)
        // Process each connection batch
        foreach ($connectionBatches as $connectionIdx => $connectionRecipients) {
            // Within each connection, split into smaller batches based on batch_size
            // (for reconnecting SMTP after every batch_size emails)
            $smtpBatches = array_chunk($connectionRecipients, $batchSize);
            
            foreach ($smtpBatches as $batch) {
                // Build HTML map for each recipient in the batch
                $htmlMap = [];
                foreach ($batch as $emailLower) {
                    $htmlMap[$emailLower] = build_tracked_html($campaignSend, $emailLower);
                }
                
                try {
                    // Send batch using single connection
                    if (isset($profile['type']) && $profile['type'] === 'api') {
                        // API sending doesn't benefit from persistent connections, send individually
                        foreach ($batch as $emailLower) {
                            $res = api_send_mail($profile, $campaignSend, $emailLower, $htmlMap[$emailLower]);
                            
                            if (!empty($res['ok']) && ($res['type'] ?? '') === 'delivered') {
                                $ins->execute([
                                    $campaignId,
                                    'delivered',
                                    json_encode([
                                        'rcpt' => $emailLower,
                                        'profile_id' => $profile['id'] ?? null,
                                        'via' => $profile['type'] ?? null,
                                        'smtp_code' => $res['code'] ?? null,
                                        'test' => $isTest ? 1 : 0,
                                    ])
                                ]);
                                $sent++;
                            } else {
                                $etype = ($res['type'] === 'deferred') ? 'deferred' : 'bounce';
                                $ins->execute([
                                    $campaignId,
                                    $etype,
                                    json_encode([
                                        'rcpt' => $emailLower,
                                        'profile_id' => $profile['id'] ?? null,
                                        'via' => $profile['type'] ?? null,
                                        'smtp_code' => $res['code'] ?? null,
                                        'smtp_msg' => $res['msg'] ?? '',
                                        'test' => $isTest ? 1 : 0,
                                    ])
                                ]);
                                
                                if ($etype === 'bounce') {
                                    add_to_unsubscribes($pdo, $emailLower);
                                }
                                $failed++;
                            }
                        }
                    } else {
                        // SMTP batch sending with persistent connection
                        $batchResult = smtp_send_batch($profile, $campaignSend, $batch, $htmlMap);
                        
                        if (isset($batchResult['results']) && is_array($batchResult['results'])) {
                            foreach ($batchResult['results'] as $recipientEmail => $res) {
                                if (!empty($res['ok']) && ($res['type'] ?? '') === 'delivered') {
                                    $ins->execute([
                                        $campaignId,
                                        'delivered',
                                        json_encode([
                                            'rcpt' => $recipientEmail,
                                            'profile_id' => $profile['id'] ?? null,
                                            'via' => 'smtp_batch',
                                            'smtp_code' => $res['code'] ?? null,
                                            'test' => $isTest ? 1 : 0,
                                        ])
                                    ]);
                                    $sent++;
                                } else {
                                    $etype = ($res['type'] === 'deferred') ? 'deferred' : 'bounce';
                                    $ins->execute([
                                        $campaignId,
                                        $etype,
                                        json_encode([
                                            'rcpt' => $recipientEmail,
                                            'profile_id' => $profile['id'] ?? null,
                                            'via' => 'smtp_batch',
                                            'smtp_code' => $res['code'] ?? null,
                                            'smtp_msg' => $res['msg'] ?? '',
                                            'test' => $isTest ? 1 : 0,
                                        ])
                                    ]);
                                    
                                    if ($etype === 'bounce') {
                                        add_to_unsubscribes($pdo, $recipientEmail);
                                    }
                                    $failed++;
                                }
                            }
                        } else {
                            // Batch failed entirely (connection error)
                            foreach ($batch as $emailLower) {
                                $ins->execute([
                                    $campaignId,
                                    'bounce',
                                    json_encode([
                                        'rcpt' => $emailLower,
                                        'error' => $batchResult['error'] ?? 'Batch connection failed',
                                        'profile_id' => $profile['id'] ?? null,
                                        'via' => 'smtp_batch',
                                        'test' => $isTest ? 1 : 0,
                                    ])
                                ]);
                                add_to_unsubscribes($pdo, $emailLower);
                                $failed++;
                            }
                        }
                    }
                } catch (Exception $e) {
                    // Handle batch exception
                    foreach ($batch as $emailLower) {
                        $ins->execute([
                            $campaignId,
                            'bounce',
                            json_encode([
                                'rcpt' => $emailLower,
                                'error' => 'Batch exception: ' . $e->getMessage(),
                                'profile_id' => $profile['id'] ?? null,
                            'test' => $isTest ? 1 : 0,
                        ])
                    ]);
                    add_to_unsubscribes($pdo, $emailLower);
                    $failed++;
                }
            }
            
            // Update progress periodically
            if (!$isTest && ($sent + $failed) % PROGRESS_UPDATE_FREQUENCY === 0) {
                update_campaign_progress($pdo, $campaignId, $sent + $failed, $total, 'sending');
            }
            } // End of smtpBatches loop
        } // End of connectionBatches loop
    } // End of if-else concurrent/sequential
    
    // Final status update - CRITICAL: This must always execute
    // Wrapped in try-finally to ensure execution even if errors occur
    try {
        if (!$isTest) {
            // Mark as completed in progress tracker
            update_campaign_progress($pdo, $campaignId, $sent + $failed, $total, 'completed');
            
            // Update main campaign status to 'sent'
            try {
                $stmt = $pdo->prepare("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id=?");
                $result = $stmt->execute([$campaignId]);
                if (!$result) {
                    error_log("Failed to update campaign status to sent for campaign ID: {$campaignId}");
                    // Force update with direct query as fallback
                    $pdo->exec("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id={$campaignId}");
                } else {
                    error_log("Successfully updated campaign {$campaignId} status to sent");
                }
            } catch (Exception $e) {
                error_log("Exception updating campaign status to sent: " . $e->getMessage());
                // Force update with direct query as last resort
                try {
                    $pdo->exec("UPDATE campaigns SET status='sent', sent_at = NOW() WHERE id={$campaignId}");
                    error_log("Fallback status update successful for campaign {$campaignId}");
                } catch (Exception $e2) {
                    error_log("Fallback status update also failed: " . $e2->getMessage());
                }
            }
        }
    } finally {
        // Ensure we always log completion regardless of any errors
        error_log("Campaign {$campaignId} send process completed: sent={$sent}, failed={$failed}, total={$total}");
    }
    
    return ['sent' => $sent, 'failed' => $failed];
}

/**
 * IMAP bounce processing — unchanged
 */
function process_imap_bounces(PDO $pdo) {
    if (!function_exists('imap_open')) {
        return;
    }

    $profiles = get_profiles($pdo);
    $ins = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");

    foreach ($profiles as $p) {
        $server = trim($p['bounce_imap_server'] ?? '');
        $user   = trim($p['bounce_imap_user'] ?? '');
        $pass   = trim($p['bounce_imap_pass'] ?? '');

        if ($server === '' || $user === '' || $pass === '') continue;

        $mailbox = $server;
        if (stripos($mailbox, '{') !== 0) {
            $hostOnly = $mailbox;
            $mailbox = "{" . $hostOnly . ":993/imap/ssl}INBOX";
        }

        try {
            $mbox = @imap_open($mailbox, $user, $pass, 0, 1);
            if (!$mbox) {
                @imap_close($mbox);
                continue;
            }

            $msgs = @imap_search($mbox, 'UNSEEN');
            if ($msgs === false) {
                $msgs = @imap_search($mbox, 'ALL');
                if ($msgs === false) $msgs = [];
            }

            foreach ($msgs as $msgno) {
                $header = @imap_headerinfo($mbox, $msgno);
                $body = @imap_body($mbox, $msgno);

                $foundEmails = [];

                if ($body) {
                    if (preg_match_all('/Final-Recipient:\s*.*?;\s*([^\s;<>"]+)/i', $body, $m1)) {
                        foreach ($m1[1] as $e) $foundEmails[] = $e;
                    }
                    if (preg_match_all('/Original-Recipient:\s*.*?;\s*([^\s;<>"]+)/i', $body, $m2)) {
                        foreach ($m2[1] as $e) $foundEmails[] = $e;
                    }
                }

                $hdrText = imap_fetchheader($mbox, $msgno);
                if ($hdrText) {
                    if (preg_match_all('/(?:To|Delivered-To|X-Original-To):\s*(.*)/i', $hdrText, $mt)) {
                        foreach ($mt[1] as $chunk) {
                            if (preg_match_all('/[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,}/i', $chunk, $emails)) {
                                foreach ($emails[0] as $e) $foundEmails[] = $e;
                            }
                        }
                    }
                }

                if ($body) {
                    if (preg_match_all('/[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,}/i', $body, $eb)) {
                        foreach ($eb[0] as $e) $foundEmails[] = $e;
                    }
                }

                $foundEmails = array_unique(array_map('strtolower', array_filter(array_map('trim', $foundEmails))));
                if (!empty($foundEmails)) {
                    foreach ($foundEmails as $rcpt) {
                        try {
                            $details = [
                                'rcpt' => $rcpt,
                                'profile_id' => $p['id'],
                                'source' => 'imap_bounce',
                                'subject' => isset($header->subject) ? (string)$header->subject : '',
                            ];
                            $ins->execute([0, 'bounce', json_encode($details)]);
                        } catch (Exception $e) {}

                        try {
                            add_to_unsubscribes($pdo, $rcpt);
                        } catch (Exception $e) {}
                    }
                }

                @imap_setflag_full($mbox, $msgno, "\\Seen");
            }

            @imap_close($mbox);
        } catch (Exception $e) {}
    }
}

///////////////////////
//  API ENDPOINT FOR REAL-TIME CAMPAIGN PROGRESS
///////////////////////
if (isset($_GET['api']) && $_GET['api'] === 'progress' && isset($_GET['campaign_id'])) {
    header('Content-Type: application/json');
    $campaignId = (int)$_GET['campaign_id'];
    $progress = get_campaign_progress($pdo, $campaignId);
    $stats = get_campaign_stats($pdo, $campaignId);
    
    echo json_encode([
        'success' => true,
        'progress' => $progress,
        'stats' => [
            'delivered' => $stats['delivered'],
            'bounce' => $stats['bounce'],
            'open' => $stats['open'],
            'click' => $stats['click'],
            'unsubscribe' => $stats['unsubscribe']
        ]
    ]);
    exit;
}

///////////////////////
//  TRACKING ENDPOINTS (open, click, unsubscribe)
///////////////////////
if (isset($_GET['t']) && in_array($_GET['t'], ['open','click','unsubscribe'], true)) {
    $t   = $_GET['t'];
    $cid = isset($_GET['cid']) ? (int)$_GET['cid'] : 0;

    if ($cid > 0) {
        $rcpt = '';
        if (!empty($_GET['r'])) {
            $rcptDecoded = base64url_decode($_GET['r']);
            if (is_string($rcptDecoded)) {
                $rcpt = $rcptDecoded;
            }
        }

        $details = [
            'rcpt' => $rcpt,
            'ip'   => $_SERVER['REMOTE_ADDR'] ?? '',
            'ua'   => $_SERVER['HTTP_USER_AGENT'] ?? '',
        ];

        $url = '';
        if ($t === 'click') {
            $uParam = $_GET['u'] ?? '';
            if ($uParam !== '') {
                $decoded = base64url_decode($uParam);
                if (is_string($decoded)) {
                    $url = $decoded;
                }
            }
            $details['url'] = $url;
        }

        $eventType = $t === 'open' ? 'open' : ($t === 'click' ? 'click' : 'unsubscribe');

        try {
            $stmt = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");
            $stmt->execute([$cid, $eventType, json_encode($details)]);
        } catch (Exception $e) {}

        if ($t === 'unsubscribe' && $rcpt !== '') {
            try {
                add_to_unsubscribes($pdo, $rcpt);
            } catch (Exception $e) { /* ignore */ }
        }

        // Special enhancement: if this is a click and we don't have an 'open' recorded for this rcpt+campaign,
        // create an estimated open event. This helps when clients block images (Yahoo/AOL etc.).
        if ($t === 'click' && $rcpt !== '') {
            try {
                if (!has_open_event_for_rcpt($pdo, $cid, $rcpt)) {
                    $estimatedOpen = [
                        'rcpt' => $rcpt,
                        'ip'   => $_SERVER['REMOTE_ADDR'] ?? '',
                        'ua'   => $_SERVER['HTTP_USER_AGENT'] ?? '',
                        'estimated' => 1,
                        'source' => 'click_fallback',
                    ];
                    $stmt2 = $pdo->prepare("INSERT INTO events (campaign_id, event_type, details) VALUES (?,?,?)");
                    $stmt2->execute([$cid, 'open', json_encode($estimatedOpen)]);
                }
            } catch (Exception $e) {}
        }
    }

    if ($t === 'open') {
        header('Content-Type: image/gif');
        echo base64_decode('R0lGODlhAQABAPAAAP///wAAACH5BAAAAAAALAAAAAABAAEAAAICRAEAOw==');
        exit;
    }

    if ($t === 'click') {
        if (!empty($url) && preg_match('~^https?://~i', $url)) {
            header('Location: ' . $url, true, 302);
        } else {
            header('Content-Type: text/plain; charset=utf-8');
            echo "OK";
        }
        exit;
    }

    if ($t === 'unsubscribe') {
        header('Content-Type: text/html; charset=utf-8');
        echo "<!doctype html><html><head><meta charset='utf-8'><title>Unsubscribed</title></head><body style='font-family:system-ui, -apple-system, sans-serif;padding:24px;'>";
        echo "<h2>Unsubscribed</h2>";
        echo "<p>You have been unsubscribed. If this was a mistake, please contact the sender.</p>";
        echo "</body></html>";
        exit;
    }
}

if (isset($_GET['ajax']) && $_GET['ajax'] === 'campaign_stats' && isset($_GET['id'])) {
    $id = (int)$_GET['id'];
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(get_campaign_stats($pdo, $id));
    exit;
}

///////////////////////
//  ACTIONS
///////////////////////
$action = $_POST['action'] ?? null;
$page   = $_GET['page'] ?? 'list';

if ($action === 'check_connection_profile') {
    $pid = (int)($_POST['profile_id'] ?? 0);
    header('Content-Type: application/json; charset=utf-8');

    if ($pid <= 0) {
        echo json_encode(['ok'=>false,'msg'=>'Invalid profile id']);
        exit;
    }

    $stmt = $pdo->prepare("SELECT * FROM sending_profiles WHERE id = ? LIMIT 1");
    $stmt->execute([$pid]);
    $profile = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$profile) {
        echo json_encode(['ok'=>false,'msg'=>'Profile not found']);
        exit;
    }

    try {
        if (($profile['type'] ?? 'smtp') === 'api') {
            $res = api_check_connection($profile);
        } else {
            $res = smtp_check_connection($profile);
        }
        echo json_encode($res);
        exit;
    } catch (Exception $e) {
        echo json_encode(['ok'=>false,'msg'=>$e->getMessage()]);
        exit;
    }
}

// AJAX endpoint for real-time campaign progress
if ($action === 'get_campaign_progress') {
    $cid = (int)($_GET['campaign_id'] ?? $_POST['campaign_id'] ?? 0);
    header('Content-Type: application/json; charset=utf-8');
    
    if ($cid <= 0) {
        echo json_encode(['ok'=>false,'error'=>'Invalid campaign ID']);
        exit;
    }
    
    try {
        $campaign = get_campaign($pdo, $cid);
        if (!$campaign) {
            echo json_encode(['ok'=>false,'error'=>'Campaign not found']);
            exit;
        }
        
        // Get event counts
        $stmt = $pdo->prepare("
            SELECT 
                SUM(CASE WHEN event_type = 'delivered' THEN 1 ELSE 0 END) as delivered,
                SUM(CASE WHEN event_type = 'bounce' THEN 1 ELSE 0 END) as bounced,
                SUM(CASE WHEN event_type = 'deferred' THEN 1 ELSE 0 END) as deferred,
                SUM(CASE WHEN event_type = 'skipped_unsubscribe' THEN 1 ELSE 0 END) as skipped,
                COUNT(*) as total_events
            FROM events 
            WHERE campaign_id = ?
            AND event_type IN ('delivered', 'bounce', 'deferred', 'skipped_unsubscribe')
        ");
        $stmt->execute([$cid]);
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get active worker count (workers that logged events in last 10 seconds)
        $stmt = $pdo->prepare("
            SELECT COUNT(DISTINCT JSON_EXTRACT(details, '$.worker_pid')) as active_workers
            FROM events 
            WHERE campaign_id = ?
            AND JSON_EXTRACT(details, '$.worker_pid') IS NOT NULL
            AND created_at >= DATE_SUB(NOW(), INTERVAL 10 SECOND)
        ");
        $stmt->execute([$cid]);
        $workerCount = (int)$stmt->fetchColumn();
        
        $response = [
            'ok' => true,
            'campaign_id' => $cid,
            'status' => $campaign['status'],
            'progress_status' => $campaign['progress_status'] ?? 'draft',
            'total_recipients' => (int)$campaign['total_recipients'],
            'progress_sent' => (int)$campaign['progress_sent'],
            'progress_total' => (int)$campaign['progress_total'],
            'delivered' => (int)($stats['delivered'] ?? 0),
            'bounced' => (int)($stats['bounced'] ?? 0),
            'deferred' => (int)($stats['deferred'] ?? 0),
            'skipped' => (int)($stats['skipped'] ?? 0),
            'total_processed' => (int)($stats['total_events'] ?? 0),
            'active_workers' => $workerCount,
            'percentage' => 0,
        ];
        
        if ($response['total_recipients'] > 0) {
            $response['percentage'] = round(($response['total_processed'] / $response['total_recipients']) * 100, 1);
        }
        
        echo json_encode($response);
        exit;
    } catch (Exception $e) {
        echo json_encode(['ok'=>false,'error'=>$e->getMessage()]);
        exit;
    }
}

if ($action === 'create_campaign') {
    $name = trim($_POST['name'] ?? '');
    if ($name === '') {
        $name = 'New Single Send';
    }
    $uuid = uuidv4();
    $stmt = $pdo->prepare("
        INSERT INTO campaigns (uuid, name, subject, preheader, status)
        VALUES (?, ?, '', '', 'draft')
    ");
    $stmt->execute([$uuid, $name]);
    $newId = (int)$pdo->lastInsertId();
    header("Location: ?page=editor&id={$newId}");
    exit;
}

if ($action === 'duplicate_campaign') {
    $cid = (int)($_POST['campaign_id'] ?? 0);
    if ($cid > 0) {
        try {
            $orig = get_campaign($pdo, $cid);
            if ($orig) {
                $newUuid = uuidv4();
                $newName = $orig['name'] . ' (Copy)';
                $stmt = $pdo->prepare("
                    INSERT INTO campaigns (uuid, name, subject, preheader, from_email, html, audience, unsubscribe_enabled, sender_name, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
                ");
                $htmlToStore = is_string($orig['html']) ? ('BASE64:' . base64_encode($orig['html'])) : '';
                $stmt->execute([
                    $newUuid,
                    $newName,
                    $orig['subject'] ?? '',
                    $orig['preheader'] ?? '',
                    $orig['from_email'] ?? '',
                    $htmlToStore,
                    $orig['audience'] ?? '',
                    $orig['unsubscribe_enabled'] ?? 0,
                    $orig['sender_name'] ?? '',
                ]);
                $newId = (int)$pdo->lastInsertId();
                header("Location: ?page=editor&id={$newId}");
                exit;
            }
        } catch (Exception $e) {}
    }
    header("Location: ?page=list");
    exit;
}

if ($action === 'bulk_campaigns') {
    $ids = $_POST['campaign_ids'] ?? [];
    $bulk_action = $_POST['bulk_action'] ?? '';
    if (!is_array($ids) || empty($ids)) {
        header("Location: ?page=list");
        exit;
    }
    $ids = array_map('intval', $ids);
    if ($bulk_action === 'delete_selected') {
        foreach ($ids as $id) {
            try {
                $stmt = $pdo->prepare("DELETE FROM events WHERE campaign_id = ?");
                $stmt->execute([$id]);
                $stmt = $pdo->prepare("DELETE FROM campaigns WHERE id = ?");
                $stmt->execute([$id]);
            } catch (Exception $e) {}
        }
    } elseif ($bulk_action === 'duplicate_selected') {
        foreach ($ids as $id) {
            try {
                $orig = get_campaign($pdo, $id);
                if ($orig) {
                    $newUuid = uuidv4();
                    $newName = $orig['name'] . ' (Copy)';
                    $stmt = $pdo->prepare("
                        INSERT INTO campaigns (uuid, name, subject, preheader, from_email, html, audience, unsubscribe_enabled, sender_name, status)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
                    ");
                    $htmlToStore = is_string($orig['html']) ? ('BASE64:' . base64_encode($orig['html'])) : '';
                    $stmt->execute([
                        $newUuid,
                        $newName,
                        $orig['subject'] ?? '',
                        $orig['preheader'] ?? '',
                        $orig['from_email'] ?? '',
                        $htmlToStore,
                        $orig['audience'] ?? '',
                        $orig['unsubscribe_enabled'] ?? 0,
                        $orig['sender_name'] ?? '',
                    ]);
                }
            } catch (Exception $e) {}
        }
    }
    header("Location: ?page=list");
    exit;
}

if ($action === 'delete_unsubscribes') {
    // Option A: delete all unsubscribes
    try {
        $pdo->beginTransaction();
        $pdo->exec("DELETE FROM unsubscribes");
        $pdo->commit();
    } catch (Exception $e) {
        try { $pdo->rollBack(); } catch (Exception $_) {}
    }
    header("Location: ?page=activity&cleared_unsubscribes=1");
    exit;
}

if ($action === 'delete_bounces') {
    // Delete all events of type 'bounce'
    try {
        $pdo->beginTransaction();
        $stmt = $pdo->prepare("DELETE FROM events WHERE event_type = ?");
        $stmt->execute(['bounce']);
        $pdo->commit();
    } catch (Exception $e) {
        try { $pdo->rollBack(); } catch (Exception $_) {}
    }
    header("Location: ?page=activity&cleared_bounces=1");
    exit;
}

if ($action === 'delete_campaign') {
    $cid = (int)($_POST['campaign_id'] ?? 0);
    if ($cid > 0) {
        try {
            $stmt = $pdo->prepare("DELETE FROM events WHERE campaign_id = ?");
            $stmt->execute([$cid]);
            $stmt = $pdo->prepare("DELETE FROM campaigns WHERE id = ?");
            $stmt->execute([$cid]);
        } catch (Exception $e) {
            // ignore
        }
    }
    header("Location: ?page=list");
    exit;
}

if ($action === 'save_campaign') {
    $id        = (int)($_POST['id'] ?? 0);
    $subject   = trim($_POST['subject'] ?? '');
    $preheader = trim($_POST['preheader'] ?? '');
    $audience  = trim($_POST['audience'] ?? '');
    $html      = $_POST['html'] ?? '';

    $rot       = get_rotation_settings($pdo);
    $rotOn     = (int)$rot['rotation_enabled'] === 1;

    if ($rotOn) {
        $existing   = get_campaign($pdo, $id);
        $fromEmail  = $existing ? $existing['from_email'] : '';
    } else {
        $fromEmail  = trim($_POST['from_email'] ?? '');
    }

    $unsubscribe_enabled = isset($_POST['unsubscribe_enabled']) ? 1 : 0;
    $sender_name = trim($_POST['sender_name'] ?? '');

    $htmlToStore = 'BASE64:' . base64_encode($html);

    try {
        $stmt = $pdo->prepare("
            UPDATE campaigns
            SET subject = ?, preheader = ?, from_email = ?, html = ?, audience = ?, unsubscribe_enabled = ?, sender_name = ?
            WHERE id = ?
        ");
        $stmt->execute([$subject, $preheader, $fromEmail, $htmlToStore, $audience, $unsubscribe_enabled, $sender_name, $id]);
    } catch (Exception $e) {
        error_log("Failed to save campaign id {$id}: " . $e->getMessage());
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) || strpos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') !== false) {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to save campaign']);
            exit;
        }
        header("Location: ?page=editor&id={$id}&save_error=1");
        exit;
    }

    if (isset($_POST['go_to_review'])) {
        header("Location: ?page=review&id={$id}");
    } else {
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) || strpos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') !== false) {
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode(['ok' => 1, 'id' => $id]);
            exit;
        }
        header("Location: ?page=editor&id={$id}&saved=1");
    }
    exit;
}

if ($action === 'send_test_message') {
    $id = (int)($_POST['id'] ?? 0);
    $addresses = trim($_POST['test_addresses'] ?? '');
    $campaign = get_campaign($pdo, $id);
    if ($campaign && $addresses !== '') {
        $toUpdate = false;
        $updFields = [];
        $updVals = [];

        if (isset($_POST['subject'])) {
            $toUpdate = true;
            $updFields[] = "subject = ?";
            $updVals[] = trim($_POST['subject']);
        }
        if (isset($_POST['preheader'])) {
            $toUpdate = true;
            $updFields[] = "preheader = ?";
            $updVals[] = trim($_POST['preheader']);
        }
        if (isset($_POST['sender_name'])) {
            $toUpdate = true;
            $updFields[] = "sender_name = ?";
            $updVals[] = trim($_POST['sender_name']);
        }
        $unsubscribe_enabled = isset($_POST['unsubscribe_enabled']) ? 1 : 0;
        if ($unsubscribe_enabled != ($campaign['unsubscribe_enabled'] ?? 0)) {
            $toUpdate = true;
            $updFields[] = "unsubscribe_enabled = ?";
            $updVals[] = $unsubscribe_enabled;
        }

        if ($toUpdate && !empty($updFields)) {
            $updVals[] = $id;
            $sql = "UPDATE campaigns SET " . implode(',', $updFields) . " WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($updVals);
            $campaign = get_campaign($pdo, $id);
        }

        $parts = array_filter(array_map('trim', preg_split("/\r\n|\n|\r|,/", $addresses)));
        $valid = [];
        foreach ($parts as $p) {
            if (filter_var($p, FILTER_VALIDATE_EMAIL)) $valid[] = $p;
        }
        if (!empty($valid)) {
            $recipientsText = implode("\n", $valid);

            session_write_close();

            if (function_exists('fastcgi_finish_request')) {
                header("Location: ?page=review&id={$id}&test_sent=1");
                echo "<!doctype html><html><body>Sending test in background... Redirecting.</body></html>";
                @ob_end_flush();
                @flush();
                fastcgi_finish_request();

                try {
                    send_campaign_real($pdo, $campaign, $recipientsText, true);
                } catch (Exception $e) {}
                exit;
            }

            // fallback: try spawn; if spawn fails do synchronous send (web request will wait)
            $spawned = spawn_background_send($pdo, $id, $recipientsText);
            if (!$spawned) {
                // synchronous fallback (guarantees it actually sends)
                ignore_user_abort(true);
                set_time_limit(0);
                try {
                    send_campaign_real($pdo, $campaign, $recipientsText, true);
                } catch (Exception $e) {}
            }
        }
    }
    header("Location: ?page=review&id={$id}&test_sent=1");
    exit;
}

if ($action === 'send_campaign') {
    $id = (int)($_POST['id'] ?? 0);
    $testRecipients = $_POST['test_recipients'] ?? '';
    $audience_select = trim($_POST['audience_select'] ?? '');
    $campaign = get_campaign($pdo, $id);
    if ($campaign) {
        $toUpdate = false;
        $updFields = [];
        $updVals = [];

        if (isset($_POST['subject'])) {
            $toUpdate = true;
            $updFields[] = "subject = ?";
            $updVals[] = trim($_POST['subject']);
        }
        if (isset($_POST['preheader'])) {
            $toUpdate = true;
            $updFields[] = "preheader = ?";
            $updVals[] = trim($_POST['preheader']);
        }
        if (isset($_POST['sender_name'])) {
            $toUpdate = true;
            $updFields[] = "sender_name = ?";
            $updVals[] = trim($_POST['sender_name']);
        }
        $unsubscribe_enabled = isset($_POST['unsubscribe_enabled']) ? 1 : 0;
        if ($unsubscribe_enabled != ($campaign['unsubscribe_enabled'] ?? 0)) {
            $toUpdate = true;
            $updFields[] = "unsubscribe_enabled = ?";
            $updVals[] = $unsubscribe_enabled;
        }

        // If rotation is OFF, allow overriding the campaign.from_email from the Review form select
        $rotSettings = get_rotation_settings($pdo);
        $rotationEnabled = (int)$rotSettings['rotation_enabled'] === 1;
        if (!$rotationEnabled && isset($_POST['from_email'])) {
            $fromOverride = trim($_POST['from_email']);
            if ($fromOverride !== '') {
                $toUpdate = true;
                $updFields[] = "from_email = ?";
                $updVals[] = $fromOverride;
                $campaign['from_email'] = $fromOverride; // keep in-memory
            }
        }

        if ($toUpdate && !empty($updFields)) {
            $updVals[] = $id;
            $sql = "UPDATE campaigns SET " . implode(',', $updFields) . " WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($updVals);
            $campaign = get_campaign($pdo, $id);
        }

        $allRecipients = [];

        if ($audience_select !== '' && strpos($audience_select, 'list:') === 0) {
            $lid = (int)substr($audience_select, strlen('list:'));
            if ($lid > 0) {
                $contacts = get_contacts_for_list($pdo, $lid);
                foreach ($contacts as $ct) {
                    $em = trim((string)($ct['email'] ?? ''));
                    if ($em === '') continue;
                    if (!filter_var($em, FILTER_VALIDATE_EMAIL)) continue;
                    if (is_unsubscribed($pdo, $em)) continue;
                    $allRecipients[] = strtolower($em);
                }
            }
        } else {
            $parts = array_filter(array_map('trim', preg_split("/\r\n|\n|\r|,/", $testRecipients)));
            foreach ($parts as $p) {
                if (!filter_var($p, FILTER_VALIDATE_EMAIL)) continue;
                if (is_unsubscribed($pdo, $p)) continue;
                $allRecipients[] = strtolower($p);
            }
        }

        $allRecipients = array_values(array_unique($allRecipients));

        if (empty($allRecipients)) {
            header("Location: ?page=review&id={$id}&no_recipients=1");
            exit;
        }

        try {
            $totalRecipients = count($allRecipients);
            $stmt = $pdo->prepare("UPDATE campaigns SET status='sending', total_recipients=? WHERE id=?");
            $stmt->execute([$totalRecipients, $campaign['id']]);
        } catch (Exception $e) {}

        $recipientsTextFull = implode("\n", $allRecipients);

        $redirect = "?page=list&sent=1&sent_campaign=" . (int)$campaign['id'];

        session_write_close();

        $rotSettings = get_rotation_settings($pdo);
        $rotationEnabled = (int)$rotSettings['rotation_enabled'] === 1;
        $activeProfiles = array_values(array_filter(get_profiles($pdo), function($p){ return (int)$p['active'] === 1; }));

        // Check if we should use multi-profile rotation or single profile
        if ($rotationEnabled && count($activeProfiles) > 1) {
            // ROTATION MODE: Use ALL active profiles and distribute emails among them
            // Use fastcgi_finish_request if available to send in background
            if (function_exists('fastcgi_finish_request')) {
                header("Location: $redirect");
                echo "<!doctype html><html><body>Sending in background... Redirecting.</body></html>";
                @ob_end_flush();
                @flush();
                fastcgi_finish_request();

                ignore_user_abort(true);
                set_time_limit(0);

                // Send using multi-profile rotation function
                try {
                    send_campaign_with_rotation($pdo, $campaign, $allRecipients, $activeProfiles, false);
                } catch (Exception $e) {
                    error_log("Campaign send error: " . $e->getMessage());
                }
                exit;
            }

            // Fallback: send synchronously (blocking)
            header("Location: $redirect");
            echo "<!doctype html><html><body>Sending campaign... Please wait.</body></html>";
            @ob_end_flush();
            @flush();

            ignore_user_abort(true);
            set_time_limit(0);

            try {
                send_campaign_with_rotation($pdo, $campaign, $allRecipients, $activeProfiles, false);
            } catch (Exception $e) {
                error_log("Campaign send error: " . $e->getMessage());
            }
            exit;
        }

        // SINGLE PROFILE MODE: Use one profile (rotation disabled or only 1 active profile)
        // Determine which profile to use
        $profile = null;
        if (!empty($activeProfiles)) {
            // No rotation or only one profile: find profile matching campaign's from_email or use first active
            foreach ($activeProfiles as $p) {
                if (strtolower($p['from_email']) === strtolower($campaign['from_email'])) {
                    $profile = $p;
                    break;
                }
            }
            if (!$profile) $profile = $activeProfiles[0];
        }

        if (!$profile) {
            header("Location: ?page=review&id={$id}&send_err=no_profile");
            exit;
        }

        // Use fastcgi_finish_request if available to send in background
        if (function_exists('fastcgi_finish_request')) {
            header("Location: $redirect");
            echo "<!doctype html><html><body>Sending in background... Redirecting.</body></html>";
            @ob_end_flush();
            @flush();
            fastcgi_finish_request();

            ignore_user_abort(true);
            set_time_limit(0);

            // Send using new connection-based function
            try {
                send_campaign_with_connections($pdo, $campaign, $allRecipients, (int)$profile['id'], false);
            } catch (Exception $e) {
                error_log("Campaign send error: " . $e->getMessage());
            }
            exit;
        }

        // Fallback: send synchronously (blocking)
        header("Location: $redirect");
        echo "<!doctype html><html><body>Sending campaign... Please wait.</body></html>";
        @ob_end_flush();
        @flush();

        ignore_user_abort(true);
        set_time_limit(0);

        try {
            send_campaign_with_connections($pdo, $campaign, $allRecipients, (int)$profile['id'], false);
        } catch (Exception $e) {
            error_log("Campaign send error: " . $e->getMessage());
        }
        exit;

        // OLD COMPLEX LOGIC REMOVED - keeping below for reference but not used
        /*
        // If rotation enabled and multiple active profiles -> batch-wise distribution per profile with parallel workers
        if ($rotationEnabled && count($activeProfiles) > 0) {
            $globalDefaultBatch = max(1, (int)($rotSettings['batch_size'] ?? 1));
            $globalTargetPerMinute = max(1, (int)($rotSettings['target_per_minute'] ?? 1000));
            $globalWorkers = max(MIN_WORKERS, min(MAX_WORKERS, (int)($rotSettings['workers'] ?? DEFAULT_WORKERS)));
            $globalMessagesPerWorker = max(MIN_MESSAGES_PER_WORKER, (int)($rotSettings['messages_per_worker'] ?? DEFAULT_MESSAGES_PER_WORKER));

            // prepare profiles meta
            $profilesMeta = [];
            foreach ($activeProfiles as $i => $p) {
                $meta = [];
                $meta['profile'] = $p;
                $meta['batch_size'] = max(1, (int)($p['batch_size'] ?? $globalDefaultBatch));
                $meta['send_rate'] = max(0, (int)($p['send_rate'] ?? 0)); // msg / second (0 = no throttle)
                $meta['max_sends'] = max(0, (int)($p['max_sends'] ?? 0)); // 0 = unlimited
                $meta['used'] = max(0, (int)($p['sends_used'] ?? 0));
                $profilesMeta[] = $meta;
            }
            $pcount = count($profilesMeta);

            // compute default per-profile send_rate if not set (distribute globalTargetPerMinute)
            $perProfilePerMin = (int)ceil($globalTargetPerMinute / max(1, $pcount));
            $defaultPerProfilePerSec = max(1, (int)ceil($perProfilePerMin / 60));

            // Distribution algorithm: sequentially assign profile.batch_size per profile, skipping profiles that reached max_sends
            $parts = array_fill(0, $pcount, []);
            $n = count($allRecipients);
            $idx = 0; // pointer into recipients
            $pi = 0;  // profile index pointer
            $roundsWithoutAssign = 0;
            while ($idx < $n) {
                $pm = &$profilesMeta[$pi];
                // remaining capacity for this profile
                if ($pm['max_sends'] > 0) {
                    $capacity = max(0, $pm['max_sends'] - $pm['used']);
                } else {
                    $capacity = PHP_INT_MAX;
                }

                if ($capacity <= 0) {
                    // skip this profile (can't accept more)
                    $pi = ($pi + 1) % $pcount;
                    $roundsWithoutAssign++;
                    if ($roundsWithoutAssign >= $pcount) {
                        // no profile can accept more sends -> break to avoid infinite loop
                        break;
                    }
                    continue;
                }

                $toTake = min($pm['batch_size'], $capacity, $n - $idx);
                $slice = array_slice($allRecipients, $idx, $toTake);
                if (!empty($slice)) {
                    $parts[$pi] = array_merge($parts[$pi], $slice);
                    $pm['used'] += count($slice);
                    $idx += count($slice);
                    $roundsWithoutAssign = 0;
                } else {
                    $roundsWithoutAssign++;
                }

                // move to next profile (respect mode)
                if (($rotSettings['mode'] ?? 'sequential') === 'random') {
                    $pi = rand(0, $pcount - 1);
                } else {
                    $pi = ($pi + 1) % $pcount;
                }
            }
            // If some recipients remain unassigned (e.g., all profiles exhausted), append them to last profile fallback
            if ($idx < $n) {
                // find any profile with capacity or fallback to first active
                $assigned = false;
                foreach ($profilesMeta as $j => $pmf) {
                    if ($pmf['max_sends'] === 0 || $pmf['max_sends'] > $pmf['used']) {
                        $parts[$j] = array_merge($parts[$j], array_slice($allRecipients, $idx));
                        $assigned = true; break;
                    }
                }
                if (!$assigned) {
                    $parts[0] = array_merge($parts[0], array_slice($allRecipients, $idx));
                }
            }

            // Now spawn per-profile workers / or send in-process. For each profile, determine effective send_rate (override)
            $spawnFailures = [];
            if (function_exists('fastcgi_finish_request')) {
                // detach response then do sends in-process sequentially (still per-profile throttled)
                header("Location: $redirect");
                echo "<!doctype html><html><body>Sending in background... Redirecting.</body></html>";
                @ob_end_flush();
                @flush();
                fastcgi_finish_request();

                ignore_user_abort(true);
                set_time_limit(0);

                foreach ($profilesMeta as $idx => $pm) {
                    $p = $pm['profile'];
                    $recips = array_values(array_unique(array_map('trim', $parts[$idx])));
                    if (empty($recips)) continue;
                    $recipsText = implode("\n", $recips);
                    $effectiveRate = $pm['send_rate'] > 0 ? $pm['send_rate'] : $defaultPerProfilePerSec;
                    $overrides = ['send_rate' => $effectiveRate];
                    try {
                        send_campaign_real($pdo, $campaign, $recipsText, false, (int)$p['id'], $overrides);
                    } catch (Exception $e) {
                        // continue
                    }
                }

                exit;
            }

            // Otherwise attempt to spawn parallel workers per profile (passing overrides so each worker can throttle)
            foreach ($profilesMeta as $idx => $pm) {
                $p = $pm['profile'];
                $recips = array_values(array_unique(array_map('trim', $parts[$idx])));
                if (empty($recips)) continue;
                $effectiveRate = $pm['send_rate'] > 0 ? $pm['send_rate'] : $defaultPerProfilePerSec;
                $overrides = ['send_rate' => $effectiveRate];
                
                // Spawn parallel workers for this profile's recipients
                $result = spawn_parallel_workers($pdo, (int)$campaign['id'], $recips, $globalWorkers, $globalMessagesPerWorker, (int)$p['id'], $overrides);
                
                if (!$result['success']) {
                    // Add failures for synchronous fallback
                    foreach ($result['failures'] as $failure) {
                        $recipsText = implode("\n", $failure['chunk']);
                        $spawnFailures[] = ['profile' => $p, 'recips' => $recipsText, 'overrides' => $overrides];
                    }
                }
            }

            // If any spawn failed, fallback to synchronous sending for those profiles (guarantees actual send)
            if (!empty($spawnFailures)) {
                // send redirect first so UI isn't waiting too long
                header("Location: $redirect");
                echo "<!doctype html><html><body>Sending (fallback) in progress... Please wait if your browser does not redirect.</body></html>";
                @ob_end_flush();
                @flush();

                ignore_user_abort(true);
                set_time_limit(0);

                foreach ($spawnFailures as $fail) {
                    try {
                        send_campaign_real($pdo, $campaign, $fail['recips'], false, (int)$fail['profile']['id'], $fail['overrides']);
                    } catch (Exception $e) {
                        // continue
                    }
                }

                exit;
            }

            // All spawns succeeded -> safe to redirect immediately
            header("Location: $redirect");
            exit;
        } else {
            // Rotation disabled: send entire campaign with the selected SMTP using parallel workers
            // Get the profile for this campaign (either forced or first active)
            $profile = null;
            if (!empty($activeProfiles)) {
                // Try to find profile matching campaign's from_email
                foreach ($activeProfiles as $p) {
                    if (strtolower($p['from_email']) === strtolower($campaign['from_email'])) {
                        $profile = $p;
                        break;
                    }
                }
                // Fallback to first active profile
                if (!$profile) $profile = $activeProfiles[0];
            }
            
            $workers = DEFAULT_WORKERS;
            $messagesPerWorker = DEFAULT_MESSAGES_PER_WORKER;
            
            if ($profile) {
                $workers = max(MIN_WORKERS, min(MAX_WORKERS, (int)($profile['workers'] ?? DEFAULT_WORKERS)));
                $messagesPerWorker = max(MIN_MESSAGES_PER_WORKER, (int)($profile['messages_per_worker'] ?? DEFAULT_MESSAGES_PER_WORKER));
            }
            
            if (function_exists('fastcgi_finish_request')) {
                header("Location: $redirect");
                echo "<!doctype html><html><body>Sending in background... Redirecting.</body></html>";
                @ob_end_flush();
                @flush();
                fastcgi_finish_request();

                try {
                    send_campaign_real($pdo, $campaign, $recipientsTextFull);
                } catch (Exception $e) {}
                exit;
            }

            // Try to spawn parallel workers
            $result = spawn_parallel_workers($pdo, (int)$campaign['id'], $allRecipients, $workers, $messagesPerWorker, $profile ? (int)$profile['id'] : null, []);
            if (!$result['success'] || $result['workers'] === 0) {
                // synchronous fallback (web request will wait) to ensure delivery instead of remaining 'sending'
                header("Location: $redirect");
                echo "<!doctype html><html><body>Sending (fallback) in progress... Please wait if your browser does not redirect.</body></html>";
                @ob_end_flush();
                @flush();
                
                ignore_user_abort(true);
                set_time_limit(0);
                try {
                    send_campaign_real($pdo, $campaign, $recipientsTextFull);
                } catch (Exception $e) {}
                exit;
            }

            // All workers spawned successfully
            header("Location: $redirect");
            exit;
        }
        */
        // END OLD COMPLEX LOGIC

    }
    header("Location: $redirect");
    exit;
}

if ($action === 'save_rotation') {
    $rotation_enabled      = isset($_POST['rotation_enabled']) ? 1 : 0;
    
    // Keep default values for backward compatibility (not shown in UI)
    $rotation_mode         = 'sequential';
    $batch_size            = 1;
    $max_sends_per_profile = 0;
    $workers               = DEFAULT_WORKERS;
    $messages_per_worker   = DEFAULT_MESSAGES_PER_WORKER;

    update_rotation_settings($pdo, [
        'rotation_enabled'      => $rotation_enabled,
        'mode'                  => $rotation_mode,
        'batch_size'            => $batch_size,
        'max_sends_per_profile' => $max_sends_per_profile,
        'workers'               => $workers,
        'messages_per_worker'   => $messages_per_worker,
    ]);

    header("Location: ?page=list&rot_saved=1");
    exit;
}

if ($action === 'save_profile') {
    $profile_id   = (int)($_POST['profile_id'] ?? 0);
    $profile_name = trim($_POST['profile_name'] ?? '');
    $type         = $_POST['type'] === 'api' ? 'api' : 'smtp';
    $from_email   = trim($_POST['from_email'] ?? '');
    $provider     = trim($_POST['provider'] ?? '');
    $host         = trim($_POST['host'] ?? '');
    $port         = (int)($_POST['port'] ?? 0);
    $username     = trim($_POST['username'] ?? '');
    $password     = trim($_POST['password'] ?? '');
    $api_key      = trim($_POST['api_key'] ?? '');
    $api_url      = trim($_POST['api_url'] ?? '');
    $headers_json = trim($_POST['headers_json'] ?? '');
    $active       = isset($_POST['active']) ? 1 : 0;

    $profile_sender_name = trim($_POST['sender_name'] ?? '');

    $bounce_server = trim($_POST['bounce_imap_server'] ?? '');
    $bounce_user = trim($_POST['bounce_imap_user'] ?? '');
    $bounce_pass = trim($_POST['bounce_imap_pass'] ?? '');
    
    // New: Connection Numbers field (MaxBulk Mailer style)
    $connection_numbers = max(MIN_CONNECTIONS, min(MAX_CONNECTIONS, (int)($_POST['connection_numbers'] ?? DEFAULT_CONNECTIONS)));
    
    // New: Batch Size field (emails per single SMTP connection)
    $batch_size = max(MIN_BATCH_SIZE, min(MAX_BATCH_SIZE, (int)($_POST['batch_size'] ?? DEFAULT_BATCH_SIZE)));
    
    // Keep old fields for backward compatibility (but set defaults, not from form)
    $random_en    = 0;
    $max_sends    = 0;
    $send_rate    = 0;
    $profile_workers = DEFAULT_WORKERS;
    $profile_messages_per_worker = DEFAULT_MESSAGES_PER_WORKER;

    try {
        if ($profile_id > 0) {
            $stmt = $pdo->prepare("
                UPDATE sending_profiles SET
                  profile_name=?, type=?, from_email=?, provider=?, host=?, port=?, username=?, password=?,
                  api_key=?, api_url=?, headers_json=?, active=?,
                  bounce_imap_server=?, bounce_imap_user=?, bounce_imap_pass=?, sender_name=?, connection_numbers=?, batch_size=?
                WHERE id=?
            ");
            $stmt->execute([
                $profile_name, $type, $from_email, $provider, $host, $port, $username, $password,
                $api_key, $api_url, $headers_json, $active,
                $bounce_server, $bounce_user, $bounce_pass, $profile_sender_name, $connection_numbers, $batch_size,
                $profile_id
            ]);
        } else {
            $stmt = $pdo->prepare("
                INSERT INTO sending_profiles
                  (profile_name, type, from_email, provider, host, port, username, password,
                   api_key, api_url, headers_json, active,
                   bounce_imap_server, bounce_imap_user, bounce_imap_pass, sender_name, connection_numbers, batch_size)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
            ");
            $stmt->execute([
                $profile_name, $type, $from_email, $provider, $host, $port, $username, $password,
                $api_key, $api_url, $headers_json, $active,
                $bounce_server, $bounce_user, $bounce_pass, $profile_sender_name, $connection_numbers, $batch_size
            ]);
        }
    } catch (Exception $e) {
        // Fallback for tables without connection_numbers column yet
        try {
            if ($profile_id > 0) {
                $stmt = $pdo->prepare("
                    UPDATE sending_profiles SET
                      profile_name=?, type=?, from_email=?, provider=?, host=?, port=?, username=?, password=?,
                      api_key=?, api_url=?, headers_json=?, active=?, sender_name=?
                    WHERE id=?
                ");
                $stmt->execute([
                    $profile_name, $type, $from_email, $provider, $host, $port, $username, $password,
                    $api_key, $api_url, $headers_json, $active, $profile_sender_name,
                    $profile_id
                ]);
            } else {
                $stmt = $pdo->prepare("
                    INSERT INTO sending_profiles
                      (profile_name, type, from_email, provider, host, port, username, password,
                       api_key, api_url, headers_json, active, sender_name)
                    VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?)
                ");
                $stmt->execute([
                    $profile_name, $type, $from_email, $provider, $host, $port, $username, $password,
                    $api_key, $api_url, $headers_json, $active, $profile_sender_name
                ]);
            }
        } catch (Exception $e2) {
            error_log("Failed to save profile: " . $e2->getMessage());
        }
    }

    header("Location: ?page=list&profiles=1");
    exit;
}

if ($action === 'delete_profile') {
    $pid = (int)($_POST['profile_id'] ?? 0);
    if ($pid > 0) {
        $stmt = $pdo->prepare("DELETE FROM sending_profiles WHERE id = ?");
        $stmt->execute([$pid]);
    }
    header("Location: ?page=list&profiles=1");
    exit;
}

///////////////////////
//  CONTACTS ACTIONS
///////////////////////
if ($action === 'create_contact_list') {
    $listName = trim($_POST['list_name'] ?? '');
    if ($listName !== '') {
        $stmt = $pdo->prepare("INSERT INTO contact_lists (name, type) VALUES (?, 'list')");
        $stmt->execute([$listName]);
    }
    header("Location: ?page=contacts");
    exit;
}

if ($action === 'delete_contact_list') {
    $lid = (int)($_POST['list_id'] ?? 0);
    if ($lid > 0) {
        $stmt = $pdo->prepare("DELETE FROM contacts WHERE list_id = ?");
        $stmt->execute([$lid]);
        $stmt = $pdo->prepare("DELETE FROM contact_lists WHERE id = ?");
        $stmt->execute([$lid]);
    }
    header("Location: ?page=contacts");
    exit;
}

if ($action === 'add_contact_manual') {
    $lid    = (int)($_POST['list_id'] ?? 0);
    $email  = trim($_POST['email'] ?? '');
    $first  = trim($_POST['first_name'] ?? '');
    $last   = trim($_POST['last_name'] ?? '');
    if ($lid > 0 && filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $stmt = $pdo->prepare("INSERT INTO contacts (list_id, email, first_name, last_name) VALUES (?, ?, ?, ?)");
        $stmt->execute([$lid, strtolower($email), $first, $last]);
    }
    header("Location: ?page=contacts&list_id=".$lid);
    exit;
}

if ($action === 'upload_contacts_csv') {
    $lid = (int)($_POST['list_id'] ?? 0);
    if ($lid > 0 && isset($_FILES['csv_file']) && is_uploaded_file($_FILES['csv_file']['tmp_name'])) {
        $path = $_FILES['csv_file']['tmp_name'];
        if (($fh = fopen($path, 'r')) !== false) {
            $header = fgetcsv($fh);
            $rows   = [];
            $emailCol = 0;

            if ($header !== false) {
                $found = null;
                foreach ($header as $i => $col) {
                    if (stripos($col, 'email') !== false) {
                        $found = $i;
                        break;
                    }
                }
                if ($found === null) {
                    $emailCol = 0;
                    $rows[] = $header;
                } else {
                    $emailCol = $found;
                }

                while (($row = fgetcsv($fh)) !== false) {
                    $rows[] = $row;
                }
            }
            fclose($fh);

            $ins = $pdo->prepare("INSERT INTO contacts (list_id, email) VALUES (?, ?)");
            foreach ($rows as $row) {
                if (!isset($row[$emailCol])) continue;
                $email = trim($row[$emailCol]);
                if ($email !== '' && filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    if (is_unsubscribed($pdo, $email)) continue;
                    $ins->execute([$lid, strtolower($email)]);
                }
            }
        }
    }
    header("Location: ?page=contacts&list_id=".$lid);
    exit;
}

///////////////////////
//  DATA FOR PAGES
///////////////////////
$rotationSettings = get_rotation_settings($pdo);
$profiles         = get_profiles($pdo);
$contactLists     = get_contact_lists($pdo);
$editProfile      = null;
if ($page === 'list' && isset($_GET['edit_profile'])) {
    $eid = (int)$_GET['edit_profile'];
    foreach ($profiles as $p) {
        if ((int)$p['id'] === $eid) {
            $editProfile = $p;
            break;
        }
    }
}

$isSingleSendsPage = in_array($page, ['list','editor','review','stats'], true);


?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>MAFIA MAILER - Professional Email Marketing</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://fonts.googleapis.com/css2?family=Open+Sans:wght@400;500;600;700&display=swap" rel="stylesheet">
  <style>
    :root {
      --sg-blue: #1A82E2;
      --sg-blue-dark: #0F5BB5;
      --sg-blue-light: #E3F2FD;
      --sg-bg: #F6F8FB;
      --sg-border: #E3E8F0;
      --sg-text: #1F2933;
      --sg-muted: #6B778C;
      --sg-success: #12B886;
      --sg-success-light: #D3F9E9;
      --sg-danger: #FA5252;
      --sg-danger-light: #FFE5E5;
      --sg-warning: #FAB005;
      --sg-warning-light: #FFF4DB;
      --sg-shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
      --sg-shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
      --sg-shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    }
    * {
      box-sizing: border-box;
    }
    html {
      scroll-behavior: smooth;
    }
    body {
      margin: 0;
      font-family: "Open Sans", system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
      background: var(--sg-bg);
      color: var(--sg-text);
      line-height: 1.6;
      -webkit-font-smoothing: antialiased;
      -moz-osx-font-smoothing: grayscale;
    }
    a { 
      color: var(--sg-blue-dark); 
      text-decoration: none;
      transition: color 0.15s ease;
    }
    a:hover { 
      text-decoration: underline;
      color: var(--sg-blue);
    }

    .app-shell {
      display: flex;
      min-height: 100vh;
      overflow: hidden;
    }

    /* MAIN LEFT NAV (SendGrid-style) */
    .nav-main {
      width: 220px;
      background: #ffffff;
      border-right: 1px solid var(--sg-border);
      display: flex;
      flex-direction: column;
    }
    .nav-header {
      padding: 16px 16px 12px;
      border-bottom: 1px solid var(--sg-border);
      display: flex;
      align-items: center;
      gap: 10px;
    }
    .nav-avatar {
      width: 32px;
      height: 32px;
      border-radius: 999px;
      background: var(--sg-blue);
      color: #fff;
      display: flex;
      align-items: center;
      justify-content: center;
      font-weight: 600;
      font-size: 14px;
    }
    .nav-user-info {
      display: flex;
      flex-direction: column;
      gap: 2px;
    }
    .nav-user-email {
      font-size: 13px;
      font-weight: 600;
    }
    .nav-user-sub {
      font-size: 11px;
      color: var(--sg-muted);
    }
    .nav-scroll {
      flex: 1;
      overflow-y: auto;
      padding: 10px 8px 16px;
    }
    .nav-section-title {
      font-size: 11px;
      font-weight: 600;
      text-transform: uppercase;
      color: var(--sg-muted);
      letter-spacing: .05em;
      margin: 10px 8px 4px;
    }
    .nav-link {
      display: flex;
      align-items: center;
      padding: 8px 12px;
      margin: 2px 4px;
      border-radius: 6px;
      font-size: 14px;
      color: var(--sg-muted);
      transition: all 0.15s ease;
      font-weight: 500;
    }
    .nav-link span {
      flex: 1;
    }
    .nav-link.active {
      background: linear-gradient(135deg, var(--sg-blue-light) 0%, #E9F3FF 100%);
      color: var(--sg-blue-dark);
      font-weight: 600;
      box-shadow: var(--sg-shadow-sm);
    }
    .nav-link:hover {
      background: var(--sg-bg);
      text-decoration: none;
      color: var(--sg-blue);
      transform: translateX(2px);
    }
    .nav-link.active:hover {
      transform: translateX(0);
    }
    .nav-footer {
      padding: 10px 12px;
      border-top: 1px solid var(--sg-border);
      font-size: 11px;
      color: var(--sg-muted);
    }

    /* Sending profiles slide panel */
    .sidebar-sp {
      position: relative;
      width: 0;
      flex-shrink: 0;
      transition: width 0.25s ease;
      overflow: hidden;
      pointer-events: none;
    }
    .sidebar-sp.open {
      width: 400px;
      pointer-events: auto;
    }
    .sidebar-inner {
      position: fixed;
      left: 220px;
      top: 0;
      bottom: 0;
      width: 400px;
      background: #fff;
      border-right: 1px solid var(--sg-border);
      box-shadow: 4px 0 16px rgba(15, 91, 181, 0.12);
      display: flex;
      flex-direction: column;
      transform: translateX(-100%);
      transition: transform 0.25s ease, visibility 0.2s ease;
      z-index: 30;
      visibility: hidden;
      pointer-events: none;
    }
    .sidebar-sp.open .sidebar-inner {
      transform: translateX(0);
      visibility: visible;
      pointer-events: auto;
    }

    .page-wrapper {
      flex: 1;
      transition: transform 0.25s ease;
      display: flex;
      flex-direction: column;
    }
    .page-wrapper.shifted {
      transform: translateX(400px);
    }

    .topbar {
      height: 64px;
      background: #fff;
      border-bottom: 1px solid var(--sg-border);
      display: flex;
      align-items: center;
      justify-content: space-between;
      padding: 0 32px;
      box-shadow: var(--sg-shadow-sm);
      /* Sticky positioning keeps topbar visible while scrolling - intentional for better navigation */
      position: sticky;
      top: 0;
      z-index: 100;
    }
    .topbar .brand {
      font-weight: 700;
      font-size: 18px;
      display: flex;
      align-items: center;
      gap: 10px;
      color: var(--sg-text);
    }
    .topbar .brand-dot {
      width: 12px;
      height: 12px;
      border-radius: 50%;
      background: linear-gradient(135deg, var(--sg-blue) 0%, var(--sg-blue-dark) 100%);
      box-shadow: 0 0 0 3px rgba(26,130,226,0.2);
      animation: pulse-dot 2s ease-in-out infinite;
    }
    @keyframes pulse-dot {
      0%, 100% {
        transform: scale(1);
      }
      50% {
        transform: scale(1.1);
      }
    }
    .topbar-actions {
      display: flex;
      align-items: center;
      gap: 8px;
    }
    .btn {
      border-radius: 6px;
      border: 1px solid transparent;
      padding: 8px 16px;
      font-size: 14px;
      font-weight: 500;
      cursor: pointer;
      display: inline-flex;
      align-items: center;
      justify-content: center;
      gap: 6px;
      background: #fff;
      color: var(--sg-text);
      transition: all 0.2s ease;
      text-decoration: none;
    }
    .btn:hover {
      text-decoration: none;
      transform: translateY(-1px);
      box-shadow: var(--sg-shadow-sm);
    }
    .btn-primary {
      background: var(--sg-blue);
      color: #fff;
      border-color: var(--sg-blue);
      box-shadow: var(--sg-shadow-sm);
    }
    .btn-primary:hover {
      background: var(--sg-blue-dark);
      border-color: var(--sg-blue-dark);
      box-shadow: var(--sg-shadow-md);
    }
    .btn-outline {
      background: #fff;
      border-color: var(--sg-border);
      color: var(--sg-text);
    }
    .btn-outline:hover {
      border-color: var(--sg-blue);
      color: var(--sg-blue);
      background: var(--sg-blue-light);
    }
    .btn-danger {
      background: var(--sg-danger);
      border-color: var(--sg-danger);
      color: #fff;
      box-shadow: var(--sg-shadow-sm);
    }
    .btn-danger:hover {
      background: #E03131;
      border-color: #E03131;
      box-shadow: var(--sg-shadow-md);
    }

    .main-content {
      padding: 32px 40px 40px;
      flex: 1;
      overflow-y: auto;
      max-width: 1400px;
      margin: 0 auto;
      width: 100%;
    }

    .page-title {
      font-size: 28px;
      font-weight: 700;
      margin-bottom: 8px;
      color: var(--sg-text);
      letter-spacing: -0.02em;
    }
    .page-subtitle {
      font-size: 14px;
      color: var(--sg-muted);
      margin-bottom: 24px;
      line-height: 1.5;
    }

    .card {
      background: #fff;
      border-radius: 8px;
      border: 1px solid var(--sg-border);
      padding: 20px 24px;
      margin-bottom: 20px;
      box-shadow: var(--sg-shadow-sm);
      transition: box-shadow 0.2s ease, transform 0.2s ease;
    }
    .card:hover {
      box-shadow: var(--sg-shadow-md);
    }
    .card-header {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-bottom: 16px;
      padding-bottom: 12px;
      border-bottom: 1px solid var(--sg-border);
    }
    .card-title {
      font-weight: 600;
      font-size: 16px;
      color: var(--sg-text);
    }
    .card-subtitle {
      font-size: 13px;
      color: var(--sg-muted);
      margin-top: 4px;
      line-height: 1.4;
    }

    .table {
      width: 100%;
      border-collapse: collapse;
      font-size: 14px;
    }
    .table th, .table td {
      padding: 14px 12px;
      border-bottom: 1px solid var(--sg-border);
      text-align: left;
    }
    .table th {
      font-size: 12px;
      text-transform: uppercase;
      letter-spacing: 0.05em;
      color: var(--sg-muted);
      font-weight: 600;
      background-color: var(--sg-bg);
    }
    .table tbody tr {
      transition: background-color 0.15s ease;
    }
    .table tbody tr:hover {
      background-color: var(--sg-bg);
    }
    .table tbody tr:last-child td {
      border-bottom: none;
    }
    .status-dot {
      width: 10px;
      height: 10px;
      border-radius: 50%;
      display: inline-block;
      margin-right: 8px;
    }
    @keyframes pulse {
      0%, 100% {
        opacity: 1;
      }
      50% {
        opacity: .7;
      }
    }
    .status-dot.status-draft { 
      background: #CED4DA; /* Grey */
    }
    .status-dot.status-queued { 
      background: #FD7E14; /* Orange */
      animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
    }
    .status-dot.status-sending { 
      background: #FCC419; /* Yellow */
      animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
    }
    .status-dot.status-sent { 
      background: var(--sg-success); /* Green */
    }

    .badge {
      display: inline-flex;
      align-items: center;
      padding: 4px 10px;
      border-radius: 999px;
      font-size: 12px;
      font-weight: 500;
      background: var(--sg-blue-light);
      color: var(--sg-blue-dark);
      border: 1px solid var(--sg-blue);
    }

    .form-row {
      display: flex;
      gap: 12px;
      margin-bottom: 12px;
    }
    .form-group {
      margin-bottom: 12px;
      flex: 1;
    }
    .form-group label {
      font-size: 12px;
      font-weight: 600;
      display: block;
      margin-bottom: 4px;
    }
    .form-group small {
      display: block;
      font-size: 11px;
      color: var(--sg-muted);
    }
    input[type="text"],
    input[type="email"],
    input[type="number"],
    input[type="password"],
    textarea,
    select {
      width: 100%;
      padding: 10px 12px;
      font-size: 14px;
      border-radius: 6px;
      border: 1px solid var(--sg-border);
      outline: none;
      transition: all 0.2s ease;
      background: #fff;
      color: var(--sg-text);
    }
    input:focus, textarea:focus, select:focus {
      border-color: var(--sg-blue);
      box-shadow: 0 0 0 3px rgba(26,130,226,0.1);
      background: #fff;
    }
    input:hover, textarea:hover, select:hover {
      border-color: var(--sg-blue);
    }
    textarea {
      min-height: 160px;
      resize: vertical;
      font-family: inherit;
    }
    input::placeholder,
    textarea::placeholder {
      color: var(--sg-muted);
    }

    .sidebar-header {
      padding: 14px 16px;
      border-bottom: 1px solid var(--sg-border);
      display: flex;
      align-items: center;
      justify-content: space-between;
    }
    .sidebar-header-title {
      font-weight: 600;
      font-size: 14px;
    }

    .sidebar-body {
      padding: 14px 16px;
      overflow-y: auto;
      flex: 1;
    }
    .sidebar-section-title {
      font-size: 12px;
      font-weight: 600;
      margin-bottom: 6px;
      color: var(--sg-muted);
      text-transform: uppercase;
      letter-spacing: .05em;
    }
    .sidebar-foot {
      padding: 10px 16px 14px;
      border-top: 1px solid var(--sg-border);
    }

    .sp-rotation-card {
      border-radius: 6px;
      border: 1px solid var(--sg-border);
      padding: 10px 10px 12px;
      background: var(--sg-bg);
      margin-bottom: 14px;
    }
    .checkbox-row {
      display: flex;
      align-items: center;
      gap: 8px;
      margin-bottom: 8px;
      font-size: 13px;
    }
    .radio-row {
      display: flex;
      gap: 12px;
      margin-bottom: 8px;
      font-size: 13px;
    }

    .profiles-list {
      display: flex;
      flex-direction: column;
      gap: 8px;
      margin-bottom: 12px;
    }
    .profile-card {
      border-radius: 8px;
      border: 1px solid var(--sg-border);
      padding: 14px 16px;
      font-size: 13px;
      background: #fff;
      transition: all 0.2s ease;
      box-shadow: var(--sg-shadow-sm);
    }
    .profile-card:hover {
      border-color: var(--sg-blue);
      box-shadow: var(--sg-shadow-md);
      transform: translateY(-2px);
    }
    .profile-card-header {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-bottom: 4px;
    }
    .profile-name {
      font-weight: 600;
      font-size: 13px;
    }
    .profile-meta {
      font-size: 11px;
      color: var(--sg-muted);
      display: flex;
      flex-wrap: wrap;
      gap: 8px;
    }
    .profile-actions {
      display: flex;
      gap: 6px;
      margin-top: 4px;
    }
    .btn-mini {
      font-size: 11px;
      padding: 2px 6px;
      border-radius: 4px;
      border: 1px solid var(--sg-border);
      background: #fff;
      cursor: pointer;
    }

    .sp-form {
      border-radius: 6px;
      border: 1px solid var(--sg-border);
      padding: 10px 10px 12px;
      margin-bottom: 10px;
    }
    .hint {
      font-size: 11px;
      color: var(--sg-muted);
    }
    .pill {
      display: inline-flex;
      align-items: center;
      padding: 2px 6px;
      border-radius: 999px;
      font-size: 11px;
      background: #EFF3F9;
      color: #4B5563;
    }

    /* Header stats (new simplified look) */
    .header-stats {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
      gap: 20px;
      background: #fff;
      border: 1px solid var(--sg-border);
      border-radius: 8px;
      padding: 24px;
      margin-bottom: 24px;
      box-shadow: var(--sg-shadow-sm);
    }
    .stat-item {
      text-align: center;
      padding: 12px;
      border-radius: 6px;
      transition: background-color 0.2s ease, transform 0.2s ease;
    }
    .stat-item:hover {
      background-color: var(--sg-bg);
      transform: translateY(-2px);
    }
    .stat-item .stat-num {
      font-size: 32px;
      font-weight: 700;
      color: var(--sg-blue-dark);
      line-height: 1;
      margin-bottom: 8px;
    }
    .stat-item .stat-label {
      font-size: 12px;
      color: var(--sg-muted);
      text-transform: uppercase;
      letter-spacing: .05em;
      font-weight: 600;
    }

    .stats-grid {
      display: grid;
      grid-template-columns: repeat(4, minmax(0, 1fr));
      gap: 12px;
      margin-bottom: 12px;
    }
    .stat-box {
      background: #fff;
      border-radius: 6px;
      border: 1px solid var(--sg-border);
      padding: 10px 12px;
    }
    .stat-label {
      font-size: 11px;
      text-transform: uppercase;
      color: var(--sg-muted);
      margin-bottom: 4px;
    }
    .stat-value {
      font-size: 18px;
      font-weight: 600;
    }
    .stat-sub {
      font-size: 11px;
      color: var(--sg-muted);
    }

    /* Contacts modal */
    .modal-backdrop {
      position: fixed;
      inset: 0;
      background: rgba(15,23,42,0.5);
      display: flex;
      justify-content: flex-end;
      z-index: 40;
      animation: fadeIn 0.2s ease;
    }
    @supports (backdrop-filter: blur(4px)) {
      .modal-backdrop {
        backdrop-filter: blur(4px);
      }
    }
    @keyframes fadeIn {
      from {
        opacity: 0;
      }
      to {
        opacity: 1;
      }
    }
    .modal-panel {
      width: 480px;
      max-width: 100%;
      background: #fff;
      padding: 28px 32px;
      box-shadow: var(--sg-shadow-lg);
      display: flex;
      flex-direction: column;
      animation: slideInRight 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }
    @keyframes slideInRight {
      from {
        transform: translateX(100%);
      }
      to {
        transform: translateX(0);
      }
    }

    /* HTML Editor modal (fixed center) */
    .html-editor-backdrop {
      display: none; /* hidden initially */
      position: fixed;
      inset: 0;
      background: rgba(15,23,42,0.6);
      display: none;
      align-items: center;
      justify-content: center;
      z-index: 80;
      animation: fadeIn 0.2s ease;
    }
    @supports (backdrop-filter: blur(4px)) {
      .html-editor-backdrop {
        backdrop-filter: blur(4px);
      }
    }
    .html-editor-backdrop.show {
      display: flex;
    }
    .html-editor-panel {
      width: 900px;
      max-width: calc(100% - 40px);
      max-height: 85vh;
      overflow: auto;
      background: #fff;
      padding: 24px;
      border-radius: 12px;
      box-shadow: var(--sg-shadow-lg);
      display: flex;
      flex-direction: column;
      gap: 12px;
      animation: scaleIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }
    @keyframes scaleIn {
      from {
        transform: scale(0.95);
        opacity: 0;
      }
      to {
        transform: scale(1);
        opacity: 1;
      }
    }
    .html-editor-panel textarea {
      min-height: 420px;
      width: 100%;
      padding: 12px;
      font-family: monospace;
      font-size: 13px;
      border: 1px solid #e6edf3;
      border-radius: 6px;
      resize: vertical;
    }
    .html-editor-actions {
      display: flex;
      gap: 8px;
      justify-content: flex-end;
    }

    .toast {
      position: fixed;
      right: 24px;
      top: 24px;
      z-index: 200;
      padding: 14px 18px;
      border-radius: 8px;
      color: #fff;
      font-weight: 600;
      font-size: 14px;
      box-shadow: var(--sg-shadow-lg);
      display: none;
      opacity: 0;
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      transform: translateY(-10px);
    }
    .toast.show { 
      display: block; 
      opacity: 1;
      transform: translateY(0);
    }
    .toast.success { 
      background: var(--sg-success);
      border: 1px solid var(--sg-success);
    }
    .toast.error { 
      background: var(--sg-danger);
      border: 1px solid var(--sg-danger);
    }

    /* ========== REVIEW PAGE ENHANCEMENTS ========== */
    /* Make the Review page resemble SendGrid layout: left details, right preview (wider preview) */
    
    /* Status indicators for concurrent mode */
    .concurrent-badge {
      display: inline-block;
      padding: 2px 6px;
      font-size: 10px;
      font-weight: 600;
      background: #4CAF50;
      color: white;
      border-radius: 3px;
      margin-left: 8px;
      text-transform: uppercase;
    }
    .concurrent-badge.sequential {
      background: #9e9e9e;
    }
    
    .review-grid {
      display: grid;
      grid-template-columns: 1fr 680px; /* bigger preview panel */
      gap: 18px;
      align-items: start;
    }
    .review-summary {
      background: #fff;
      border-radius: 6px;
      border: 1px solid var(--sg-border);
      padding: 16px;
    }
    .review-row {
      display: flex;
      justify-content: space-between;
      padding: 10px 6px;
      border-bottom: 1px solid #f1f5f9;
    }
    .review-row .left { color: var(--sg-muted); font-weight:600; width:40%; }
    .review-row .right { width:58%; text-align:right; color:var(--sg-text); }

    .test-send-card {
      margin-top: 14px;
      background: #fff;
      border-radius: 6px;
      border: 1px solid var(--sg-border);
      padding: 14px;
    }
    .test-send-card small.hint { display:block; margin-top:8px; color:var(--sg-muted); }

    .preview-box {
      border:1px solid var(--sg-border);
      border-radius:6px;
      background:#fff;
      padding:10px;
      min-height:640px; /* taller preview */
      overflow:auto;
    }

    /*
      ============================
      NEW / FIXED STYLES FOR EDITOR
      These styles make the left "modules" area show tiles (like Image 2)
      and style the canvas placeholder and blocks for a nicer visual layout.
      ============================
    */
    .editor-shell {
      display: flex;
      gap: 18px;
      align-items: flex-start;
    }
    .editor-left {
      width: 320px; /* make left sidebar a fixed column like SendGrid */
      min-width: 260px;
      max-width: 360px;
      background: transparent;
      padding: 12px;
    }

    /* Modules grid: tiles with icons */
    .modules-grid {
      display: grid;
      grid-template-columns: repeat(2, minmax(0, 1fr));
      gap: 12px;
      align-items: start;
      margin-bottom: 10px;
    }
    .module-tile {
      background: #fff;
      border: 1px solid var(--sg-border);
      border-radius: 6px;
      padding: 18px 12px;
      text-align: center;
      cursor: pointer;
      color: var(--sg-muted);
      display: flex;
      flex-direction: column;
      align-items: center;
      gap: 8px;
      min-height: 84px;
      transition: box-shadow .15s ease, border-color .15s ease, transform .08s ease;
    }
    .module-tile .icon {
      font-size: 22px;
      color: var(--sg-blue);
      line-height: 1;
    }
    .module-tile:hover {
      border-color: var(--sg-blue);
      box-shadow: 0 6px 16px rgba(26,130,226,0.08);
      color: var(--sg-text);
      transform: translateY(-2px);
    }
    .modules-pane .sidebar-section-title {
      margin-top: 6px;
      margin-bottom: 12px;
    }

    /* Canvas and placeholder */
    .editor-right {
      flex: 1;
      display: flex;
      flex-direction: column;
      gap: 12px;
      padding-left: 12px;
    }
    .canvas-area {
      border-radius: 6px;
      border: 1px solid var(--sg-border);
      background: #fff;
      min-height: 420px;
      display: flex;
      align-items: center;
      justify-content: center;
      padding: 30px;
      position: relative;
      overflow: auto;
    }
    .drag-placeholder {
      background: #FBFCFD;
      border: 1px dashed #EEF2F6;
      padding: 60px 40px;
      color: #9AA6B2;
      border-radius: 6px;
      width: 70%;
      text-align: center;
      font-size: 15px;
    }

    /* Canvas block wrapper */
    .canvas-block {
      background: #fff;
      border: 1px solid #f1f5f9;
      border-radius: 6px;
      padding: 12px;
      margin-bottom: 12px;
      width: 100%;
      position: relative;
      box-shadow: none;
    }
    .canvas-block + .canvas-block {
      margin-top: 12px;
    }
    .block-remove {
      position: absolute;
      top: 8px;
      right: 8px;
      background: #fff;
      border: 1px solid var(--sg-border);
      border-radius: 4px;
      width: 28px;
      height: 28px;
      display: inline-flex;
      align-items: center;
      justify-content: center;
      cursor: pointer;
      font-size: 14px;
      color: #666;
    }
    .block-remove:hover {
      background: #fff;
      border-color: var(--sg-blue);
      color: var(--sg-blue-dark);
    }
    .block-content {
      min-height: 40px;
    }

    /* Code module visual improvements */
    .code-module {
      border: 1px dashed #eef2f6;
      border-radius: 6px;
      overflow: hidden;
    }
    .code-module-header {
      background: #1f6f9f;
      color: #fff;
      padding: 8px 10px;
      display: flex;
      justify-content: space-between;
      align-items: center;
    }
    .code-module-body {
      padding: 12px;
      background: #fff;
      color: #333;
    }
    .code-placeholder {
      color: #9aa6b2;
      font-style: normal;
    }

    /* Responsive design improvements */
    @media (max-width: 1400px) {
      .review-grid { grid-template-columns: 1fr 520px; }
      .main-content {
        padding: 28px 32px 32px;
      }
    }

    @media (max-width: 1200px) {
      .editor-left { width: 300px; height: auto; }
      .drag-placeholder { width: 100%; }
      .canvas-block, .drag-placeholder { width: 100%; }
      .modules-grid { grid-template-columns: repeat(1, minmax(0,1fr)); }
      .header-stats {
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 16px;
        padding: 20px;
      }
    }

    @media (max-width: 900px) {
      .form-row {
        flex-direction: column;
      }
      .stats-grid {
        grid-template-columns: repeat(2, minmax(0, 1fr));
      }
      .header-stats {
        grid-template-columns: repeat(2, 1fr);
        gap: 12px;
        padding: 16px;
      }
      .stat-item .stat-num {
        font-size: 24px;
      }
      .nav-main {
        display: none;
      }
      .sidebar-inner {
        left: 0;
        width: 100%;
      }
      .sidebar-sp.open {
        width: 100%;
      }
      .page-wrapper.shifted {
        transform: none;
      }
      .editor-shell {
        flex-direction: column;
      }
      .editor-left, .editor-right { width: 100%; }
      .review-grid { grid-template-columns: 1fr; }
      .preview-box { min-height:420px; }
      .main-content {
        padding: 20px 16px 24px;
      }
      .page-title {
        font-size: 24px;
      }
      .card {
        padding: 16px 18px;
      }
    }
    
    @media (max-width: 600px) {
      .header-stats {
        grid-template-columns: 1fr;
      }
      .card-header .btn {
        width: 100%;
        justify-content: center;
      }
      .card-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 12px;
      }
      .page-title {
        font-size: 20px;
      }
    }
  </style>
</head>
<body>
<div class="app-shell">

  <!-- LEFT NAV (SendGrid-style) -->
  <aside class="nav-main">
    <div class="nav-header">
      <div class="nav-avatar"><?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?></div>
      <div class="nav-user-info">
        <div class="nav-user-email"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></div>
        <div class="nav-user-sub">MAFIA MAILER</div>
      </div>
    </div>
    <div class="nav-scroll">
      <div class="nav-section-title">Email Marketing</div>

      <div class="nav-section-title">Marketing</div>
      <a href="?page=list" class="nav-link <?php echo $isSingleSendsPage ? 'active' : ''; ?>">
        <span>Single Sends</span>
      </a>
      <a href="?page=contacts" class="nav-link <?php echo $isContactsPage ? 'active' : ''; ?>">
        <span>Contacts</span>
      </a>
      <a href="#" class="nav-link"><span>Design Library</span></a>

      <div class="nav-section-title">Analytics</div>
      <a href="?page=stats" class="nav-link"><span>Stats</span></a>
      <a href="?page=activity" class="nav-link"><span>Activity</span></a>
    </div>
    <div class="nav-footer">
      <a href="?action=logout" style="color: var(--sg-danger); display: block; padding: 8px 0; font-weight: 600;">🚪 Logout</a>
      MAFIA MAILER v1.0<br>Professional Email Marketing
    </div>
  </aside>

  <!-- LEFT SLIDE PANEL: Sending Profiles -->
  <div class="sidebar-sp" id="spSidebar">
    <div class="sidebar-inner">
      <div class="sidebar-header">
        <div class="sidebar-header-title">Sending Profiles</div>
        <button class="btn btn-outline" type="button" id="spCloseBtn">✕</button>
      </div>
      <div class="sidebar-body">
        <!-- Rotation settings -->
        <form method="post" class="sp-rotation-card">
          <input type="hidden" name="action" value="save_rotation">
          <div class="sidebar-section-title">Rotation Settings</div>
          <div class="checkbox-row">
            <input type="checkbox" name="rotation_enabled" id="rot_enabled" <?php if ($rotationSettings['rotation_enabled']) echo 'checked'; ?>>
            <label for="rot_enabled">Enable Global SMTP/API Rotation</label>
          </div>
          <div class="form-group">
            <small class="hint" style="display:block; margin-top:8px;">
              When enabled, campaigns will rotate through all active sending profiles. 
              Configure individual profiles with Connection Numbers to control sending speed.
              <?php if (function_exists('pcntl_fork')): ?>
                <br><strong style="color:#4CAF50;">✓ Concurrent Mode Available:</strong> Multiple connections will send in parallel for maximum speed.
              <?php else: ?>
                <br><strong style="color:#ff9800;">⚠ Sequential Mode:</strong> PHP pcntl extension not available. Connections will process sequentially.
              <?php endif; ?>
            </small>
          </div>
          <button class="btn btn-primary" type="submit">Save Rotation</button>
        </form>

        <!-- Profiles list -->
        <div class="profiles-list">
          <div class="sidebar-section-title">Profiles</div>
          <?php if (empty($profiles)): ?>
            <div class="hint">No profiles yet. Create your first SMTP or API profile below.</div>
          <?php else: ?>
            <?php foreach ($profiles as $p): ?>
              <div class="profile-card" id="profile-card-<?php echo (int)$p['id']; ?>">
                <div class="profile-card-header">
                  <div class="profile-name"><?php echo h($p['profile_name']); ?></div>
                  <div class="pill"><?php echo strtoupper($p['type']); ?> · <?php echo h($p['provider']); ?></div>
                </div>
                <div class="profile-meta">
                  <span>From: <?php echo h($p['from_email']); ?></span>
                  <?php if (!empty($p['sender_name'])): ?>
                    <span>Sender: <?php echo h($p['sender_name']); ?></span>
                  <?php endif; ?>
                  <span>Status: <?php echo $p['active'] ? 'Active' : 'Disabled'; ?></span>
                  <?php if (!empty($p['connection_numbers'])): ?>
                    <span>Connections: <?php echo (int)$p['connection_numbers']; ?></span>
                  <?php endif; ?>
                  <?php if (!empty($p['batch_size'])): ?>
                    <span>Batch Size: <?php echo (int)$p['batch_size']; ?> emails/connection</span>
                  <?php endif; ?>
                </div>
                <div class="profile-actions">
                  <a href="?page=list&edit_profile=<?php echo (int)$p['id']; ?>" class="btn-mini">Edit</a>
                  <button type="button" class="btn-mini check-conn-btn" data-pid="<?php echo (int)$p['id']; ?>">Check Connection</button>
                  <form method="post" style="display:inline;">
                    <input type="hidden" name="action" value="delete_profile">
                    <input type="hidden" name="profile_id" value="<?php echo (int)$p['id']; ?>">
                    <button type="submit" class="btn-mini" onclick="return confirm('Delete this profile?');">Delete</button>
                  </form>
                </div>
                <div style="margin-top:8px; font-size:12px; color:var(--sg-muted);" id="profile-conn-status-<?php echo (int)$p['id']; ?>"></div>
              </div>
            <?php endforeach; ?>
          <?php endif; ?>
        </div>

        <!-- Add / Edit profile -->
        <div class="sidebar-section-title">
          <?php echo $editProfile ? 'Edit Profile' : 'Add New Profile'; ?>
        </div>
        <form method="post" class="sp-form" id="profileForm">
          <input type="hidden" name="action" value="save_profile">
          <input type="hidden" name="profile_id" value="<?php echo $editProfile ? (int)$editProfile['id'] : 0; ?>">
          <div class="form-group">
            <label>Profile Name</label>
            <input type="text" name="profile_name" required value="<?php echo $editProfile ? h($editProfile['profile_name']) : ''; ?>">
          </div>
          <div class="form-group">
            <label>Profile Type</label>
            <select name="type" id="pf_type">
              <option value="smtp" <?php if(!$editProfile || $editProfile['type']==='smtp') echo 'selected'; ?>>SMTP</option>
              <option value="api"  <?php if($editProfile && $editProfile['type']==='api') echo 'selected'; ?>>API</option>
            </select>
          </div>
          <div class="form-group">
            <label>From Email</label>
            <input type="email" name="from_email" required value="<?php echo $editProfile ? h($editProfile['from_email']) : ''; ?>">
          </div>

          <!-- NEW: Sender name per profile -->
          <div class="form-group">
            <label>Sender Name (display name)</label>
            <input type="text" name="sender_name" placeholder="e.g. Acme Co." value="<?php echo $editProfile ? h($editProfile['sender_name'] ?? '') : ''; ?>">
            <small class="hint">This name will be used in the From header when rotation is enabled (or as fallback).</small>
          </div>

          <div class="form-group">
            <label>Provider</label>
            <select name="provider" id="pf_provider">
              <?php
                $providers = ['Gmail','Outlook','Yahoo','Zoho','SendGrid SMTP','MailGun','MailJet','Amazon SES','SendGrid API','MailJet API','Custom'];
                $selectedProvider = $editProfile ? $editProfile['provider'] : '';
                foreach ($providers as $prov):
              ?>
                <option value="<?php echo h($prov); ?>" <?php if ($selectedProvider === $prov) echo 'selected'; ?>>
                  <?php echo h($prov); ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>

          <div id="pf_smtp_fields" style="<?php echo (!$editProfile || $editProfile['type']==='smtp') ? '' : 'display:none'; ?>">
            <div class="form-group">
              <label>Host</label>
              <input type="text" name="host" value="<?php echo $editProfile ? h($editProfile['host']) : ''; ?>">
            </div>
            <div class="form-row">
              <div class="form-group">
                <label>Port</label>
                <input type="number" name="port" value="<?php echo $editProfile ? (int)$editProfile['port'] : 587; ?>">
              </div>
              <div class="form-group">
                <label>Username</label>
                <input type="text" name="username" value="<?php echo $editProfile ? h($editProfile['username']) : ''; ?>">
              </div>
            </div>
            <div class="form-group">
              <label>Password</label>
              <input type="text" name="password" value="<?php echo $editProfile ? h($editProfile['password']) : ''; ?>">
            </div>

            <div style="margin-top:8px; border-top:1px dashed var(--sg-border); padding-top:8px;">
              <div class="form-group">
                <label>Bounce IMAP Server (optional)</label>
                <input type="text" name="bounce_imap_server" placeholder="e.g. {imap.example.com:993/imap/ssl}INBOX" value="<?php echo $editProfile ? h($editProfile['bounce_imap_server'] ?? '') : ''; ?>">
                <small class="hint">Provide full IMAP mailbox string (or host) to scan bounces. Leave blank to skip.</small>
              </div>
              <div class="form-row">
                <div class="form-group">
                  <label>Bounce IMAP User</label>
                  <input type="text" name="bounce_imap_user" value="<?php echo $editProfile ? h($editProfile['bounce_imap_user'] ?? '') : ''; ?>">
                </div>
                <div class="form-group">
                  <label>Bounce IMAP Pass</label>
                  <input type="text" name="bounce_imap_pass" value="<?php echo $editProfile ? h($editProfile['bounce_imap_pass'] ?? '') : ''; ?>">
                </div>
              </div>
            </div>
          </div>

          <div id="pf_api_fields" style="<?php echo ($editProfile && $editProfile['type']==='api') ? '' : 'display:none'; ?>">
            <div class="form-group">
              <label>API Key</label>
              <input type="text" name="api_key" value="<?php echo $editProfile ? h($editProfile['api_key']) : ''; ?>">
            </div>
            <div class="form-group">
              <label>API URL</label>
              <input type="text" name="api_url" value="<?php echo $editProfile ? h($editProfile['api_url']) : ''; ?>">
            </div>
            <div class="form-group">
              <label>Headers JSON (optional)</label>
              <textarea name="headers_json" rows="3"><?php echo $editProfile ? h($editProfile['headers_json']) : ''; ?></textarea>
            </div>
          </div>

          <div class="form-row">
            <div class="form-group">
              <label>Connection Numbers</label>
              <input type="number" name="connection_numbers" min="<?php echo MIN_CONNECTIONS; ?>" max="<?php echo MAX_CONNECTIONS; ?>" value="<?php echo $editProfile ? (int)($editProfile['connection_numbers'] ?? DEFAULT_CONNECTIONS) : DEFAULT_CONNECTIONS; ?>">
              <small class="hint">Number of concurrent SMTP connections (1-40, recommended: 5). Controls sending speed like MaxBulk Mailer.</small>
            </div>
          </div>

          <div class="form-row">
            <div class="form-group">
              <label>Batch Size (Emails per Connection)</label>
              <input type="number" name="batch_size" min="<?php echo MIN_BATCH_SIZE; ?>" max="<?php echo MAX_BATCH_SIZE; ?>" value="<?php echo $editProfile ? (int)($editProfile['batch_size'] ?? DEFAULT_BATCH_SIZE) : DEFAULT_BATCH_SIZE; ?>">
              <small class="hint">Number of emails to send through a single SMTP connection (1-500, recommended: 50). Higher values improve performance but may hit server limits.</small>
            </div>
          </div>

          <div class="checkbox-row">
            <input type="checkbox" name="active" id="pf_active" <?php if(!$editProfile || $editProfile['active']) echo 'checked'; ?>>
            <label for="pf_active">Active</label>
          </div>

          <button class="btn btn-primary" type="submit"><?php echo $editProfile ? 'Save Changes' : 'Create Profile'; ?></button>
        </form>
      </div>
      <div class="sidebar-foot">
        <div class="hint">Rotation + profiles apply globally to all Single Sends. Configure bounce mailbox to auto-add bounces to unsubscribes.</div>
      </div>
    </div>
  </div>

  <!-- MAIN PAGE -->
  <div class="page-wrapper" id="pageWrapper">
    <div class="topbar">
      <div class="brand">
        <div class="brand-dot"></div>
        <?php echo BRAND_NAME; ?>
      </div>
      <div class="topbar-actions">
        <?php if ($page === 'list'): ?>
          <a href="?page=contacts" class="btn btn-outline">Contacts</a>
          <button class="btn btn-outline" id="spOpenBtn" type="button">Sending Profiles ⚙</button>
        <?php elseif ($page === 'contacts'): ?>
          <a href="?page=list" class="btn btn-outline">Single Sends</a>
        <?php else: ?>
          <a href="?page=list" class="btn btn-outline">Single Sends</a>
          <a href="?page=contacts" class="btn btn-outline">Contacts</a>
        <?php endif; ?>
      </div>
    </div>

    <div class="main-content">
      <?php if ($page === 'list'): ?>
        <?php
          $stmt = $pdo->query("SELECT * FROM campaigns ORDER BY created_at DESC");
          $campaigns = $stmt->fetchAll(PDO::FETCH_ASSOC);
        ?>
        <div class="page-title">Single Sends</div>
        <div class="page-subtitle">Design, review, and send one-off campaigns, just like SendGrid.</div>

        <div class="card">
          <div class="card-header">
            <div>
              <div class="card-title">Campaigns</div>
              <div class="card-subtitle">Draft and sent Single Sends with live stats.</div>
            </div>
            <form method="post" style="display:flex; gap:8px; align-items:center;">
              <input type="hidden" name="action" value="create_campaign">
              <input type="text" name="name" placeholder="Campaign name" style="font-size:13px; padding:6px 8px;">
              <button type="submit" class="btn btn-primary">+ Create Single Send</button>
            </form>
          </div>

          <!-- Bulk actions toolbar -->
          <div style="display:flex; gap:8px; align-items:center; margin-bottom:8px;">
            <form method="post" id="bulkActionsForm" style="display:flex; gap:8px; align-items:center;">
              <input type="hidden" name="action" value="bulk_campaigns">
              <select name="bulk_action" id="bulkActionSelect" style="padding:6px;">
                <option value="">Bulk actions</option>
                <option value="delete_selected">Delete selected</option>
                <option value="duplicate_selected">Duplicate selected</option>
              </select>
              <button type="submit" class="btn btn-outline">Apply</button>
            </form>

            <div style="margin-left:auto; display:flex; gap:8px;">
              <form method="post" onsubmit="return confirm('Delete ALL unsubscribes? This will remove every address from unsubscribes table.');">
                <input type="hidden" name="action" value="delete_unsubscribes">
                <button class="btn btn-outline" type="submit">Clear Unsubscribes</button>
              </form>
              <form method="post" onsubmit="return confirm('Delete ALL bounce events? This will remove all bounce events from events table.');">
                <input type="hidden" name="action" value="delete_bounces">
                <button class="btn btn-outline" type="submit">Clear Bounces</button>
              </form>
            </div>
          </div>

          <form method="post" id="campaignsTableForm">
            <input type="hidden" name="action" value="bulk_campaigns">
            <table class="table">
              <thead>
                <tr>
                  <th style="width:36px;"><input type="checkbox" id="chkAll"></th>
                  <th>Campaign</th>
                  <th>Status</th>
                  <th>Subject</th>
                  <th>Sent</th>
                  <th>Target</th>
                  <th>Bounces</th>
                  <th>Delivered</th>
                  <th>Opens</th>
                  <th>Clicks</th>
                  <th>Updated</th>
                  <th>Actions</th>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($campaigns)): ?>
                  <tr>
                    <td colspan="12" style="text-align:center; padding:20px; color:var(--sg-muted);">
                      No campaigns yet. Create your first Single Send above.
                    </td>
                  </tr>
                                 <?php else: ?>
                  <?php foreach ($campaigns as $c):
                    $stats = get_campaign_stats($pdo, (int)$c['id']);
                    $status = $c['status'];
                    // Support queued status: Grey (draft) → Orange (queued) → Yellow (sending) → Green (sent)
                    if ($status === 'sent') {
                        $dotClass = 'status-sent';
                    } elseif ($status === 'sending') {
                        $dotClass = 'status-sending';
                    } elseif ($status === 'queued') {
                        $dotClass = 'status-queued';
                    } else {
                        $dotClass = 'status-draft';
                    }
                    $link = $status === 'sent' ? '?page=stats&id='.$c['id'] : '?page=editor&id='.$c['id'];
                  ?>
                    <tr data-cid="<?php echo (int)$c['id']; ?>">
                      <td><input type="checkbox" name="campaign_ids[]" value="<?php echo (int)$c['id']; ?>" class="campaign-checkbox"></td>
                      <td>
                        <a href="<?php echo $link; ?>">
                          <span class="status-dot <?php echo $dotClass; ?>"></span>
                          <?php echo h($c['name']); ?>
                        </a>
                      </td>
                      <td><?php echo ucfirst($status); ?>
                        <?php if ($status === 'sending' || $status === 'queued'): ?>
                          <div id="progress-bar-<?php echo (int)$c['id']; ?>" style="margin-top:4px;">
                            <div style="background:#eee; height:4px; border-radius:2px; overflow:hidden;">
                              <div id="progress-fill-<?php echo (int)$c['id']; ?>" style="background:#4CAF50; height:100%; width:0%; transition:width 0.3s;"></div>
                            </div>
                            <div id="progress-text-<?php echo (int)$c['id']; ?>" style="font-size:11px; color:var(--sg-muted); margin-top:2px;">
                              0% • 0 workers
                            </div>
                          </div>
                        <?php endif; ?>
                      </td>
                      <td><?php echo h($c['subject']); ?></td>
                      <td><?php echo $c['sent_at'] ? h($c['sent_at']) : '—'; ?></td>
                      <td><?php echo (int)$stats['target']; ?></td>
                      <td><span id="bounce-<?php echo (int)$c['id']; ?>"><?php echo (int)$stats['bounce']; ?></span></td>
                      <td><span id="delivered-<?php echo (int)$c['id']; ?>"><?php echo (int)$stats['delivered']; ?></span></td>
                      <td><span id="open-<?php echo (int)$c['id']; ?>"><?php echo (int)$stats['open']; ?></span></td>
                      <td><span id="click-<?php echo (int)$c['id']; ?>"><?php echo (int)$stats['click']; ?></span></td>
                      <td><?php echo h($c['updated_at']); ?></td>
                      <td>
                        <a class="btn-mini" href="<?php echo $link; ?>">Open</a>
                        <form method="post" style="display:inline;">
                          <input type="hidden" name="action" value="duplicate_campaign">
                          <input type="hidden" name="campaign_id" value="<?php echo (int)$c['id']; ?>">
                          <button type="submit" class="btn-mini">Duplicate</button>
                        </form>
                        <form method="post" style="display:inline;">
                          <input type="hidden" name="action" value="delete_campaign">
                          <input type="hidden" name="campaign_id" value="<?php echo (int)$c['id']; ?>">
                          <button type="submit" class="btn-mini" onclick="return confirm('Delete this campaign and its events?');">Delete</button>
                        </form>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </form>
        </div>

        <script>
          (function(){
            // If URL contains sent_campaign=<id>, start polling its stats to show incremental delivered updates
            function getQueryParam(name) {
              var params = new URLSearchParams(window.location.search);
              return params.get(name);
            }
            var sentCampaign = getQueryParam('sent_campaign');
            if (sentCampaign) {
              var cid = sentCampaign;
              var interval = 2000;
              var timer = setInterval(function(){
                fetch(window.location.pathname + '?ajax=campaign_stats&id=' + encodeURIComponent(cid))
                  .then(function(r){ return r.json(); })
                  .then(function(data){
                    if (!data) return;
                    var d = document.getElementById('delivered-' + cid);
                    var o = document.getElementById('open-' + cid);
                    var c = document.getElementById('click-' + cid);
                    if (d) d.innerText = data.delivered || 0;
                    if (o) o.innerText = data.open || 0;
                    if (c) c.innerText = data.click || 0;
                  }).catch(function(){});
              }, interval);
              // stop polling after 10 minutes as safety
              setTimeout(function(){ clearInterval(timer); }, 10 * 60 * 1000);
            }

            // Real-time progress polling for sending/queued campaigns
            var sendingCampaigns = document.querySelectorAll('[data-cid]');
            var progressPollers = {};
            
            sendingCampaigns.forEach(function(row){
              var cid = row.getAttribute('data-cid');
              var statusDot = row.querySelector('.status-dot');
              
              if (statusDot && (statusDot.classList.contains('status-sending') || statusDot.classList.contains('status-queued'))) {
                // Start polling progress for this campaign
                progressPollers[cid] = setInterval(function(){
                  var fd = new FormData();
                  fd.append('action', 'get_campaign_progress');
                  fd.append('campaign_id', cid);
                  
                  fetch(window.location.pathname, {
                    method: 'POST',
                    body: fd,
                    credentials: 'same-origin'
                  })
                  .then(function(r){ return r.json(); })
                  .then(function(data){
                    if (!data || !data.ok) return;
                    
                    // Update progress bar
                    var fillEl = document.getElementById('progress-fill-' + cid);
                    var textEl = document.getElementById('progress-text-' + cid);
                    var deliveredEl = document.getElementById('delivered-' + cid);
                    var bounceEl = document.getElementById('bounce-' + cid);
                    
                    if (fillEl) {
                      fillEl.style.width = data.percentage + '%';
                    }
                    if (textEl) {
                      var workersText = data.active_workers > 0 ? data.active_workers + ' workers' : 'processing';
                      textEl.innerText = data.percentage + '% • ' + data.total_processed + '/' + data.total_recipients + ' • ' + workersText;
                    }
                    if (deliveredEl) {
                      deliveredEl.innerText = data.delivered || 0;
                    }
                    if (bounceEl) {
                      bounceEl.innerText = data.bounced || 0;
                    }
                    
                    // If campaign finished, stop polling and reload page
                    // Check multiple conditions to ensure campaign is truly complete
                    var isProcessingComplete = data.progress_status === 'completed' ||
                                             (data.total_recipients > 0 && data.total_processed >= data.total_recipients);
                    
                    if (data.status === 'sent') {
                      // Status is already 'sent', safe to reload immediately
                      clearInterval(progressPollers[cid]);
                      delete progressPollers[cid];
                      
                      setTimeout(function(){
                        window.location.reload();
                      }, 1000);
                    } else if (isProcessingComplete) {
                      // Processing is complete but status not yet updated
                      // Start waiting timer if not already started
                      if (!progressPollers[cid + '_waiting']) {
                        progressPollers[cid + '_waiting'] = Date.now();
                        console.log('Campaign ' + cid + ' processing complete, waiting for status update...');
                      }
                      
                      // If we've been waiting more than 10 seconds, reload anyway
                      var waitTime = Date.now() - progressPollers[cid + '_waiting'];
                      if (waitTime > 10000) {
                        console.log('Campaign ' + cid + ' timeout waiting for status, reloading...');
                        clearInterval(progressPollers[cid]);
                        delete progressPollers[cid];
                        delete progressPollers[cid + '_waiting'];
                        window.location.reload();
                      }
                    }
                  })
                  .catch(function(err){
                    console.error('Progress poll error:', err);
                  });
                }, 2000); // Poll every 2 seconds
                
                // Stop polling after 20 minutes as safety
                setTimeout(function(){
                  if (progressPollers[cid]) {
                    clearInterval(progressPollers[cid]);
                    delete progressPollers[cid];
                  }
                }, 20 * 60 * 1000);
              }
            });

            // select all checkbox
            var chkAll = document.getElementById('chkAll');
            if (chkAll) {
              chkAll.addEventListener('change', function(){
                var boxes = document.querySelectorAll('.campaign-checkbox');
                boxes.forEach(function(b){ b.checked = chkAll.checked; });
              });
            }

            // bulk actions form should copy selected ids into its own form when submitted
            var bulkForm = document.getElementById('bulkActionsForm');
            bulkForm.addEventListener('submit', function(e){
              var sel = document.querySelectorAll('.campaign-checkbox:checked');
              if (sel.length === 0) {
                alert('No campaigns selected.');
                e.preventDefault();
                return false;
              }
              // create hidden inputs for each selected id
              // remove previous if any
              Array.from(bulkForm.querySelectorAll('input[name="campaign_ids[]"]')).forEach(function(n){ n.remove(); });
              sel.forEach(function(cb){
                var inp = document.createElement('input');
                inp.type = 'hidden';
                inp.name = 'campaign_ids[]';
                inp.value = cb.value;
                bulkForm.appendChild(inp);
              });
            });
          })();
        </script>

      <?php elseif ($page === 'editor'): ?>
        <?php
          $id = (int)($_GET['id'] ?? 0);
          $campaign = get_campaign($pdo, $id);
          if (!$campaign) {
            echo "<p>Campaign not found.</p>";
          } else {
            $rotOnEditor = (int)$rotationSettings['rotation_enabled'] === 1;
        ?>
          <div class="page-title">Editor — <?php echo h($campaign['name']); ?></div>
          <div class="page-subtitle">Define subject, from, preheader and design your email content.</div>

          <!-- Editor shell: left modules + envelope, right canvas -->
          <form method="post" id="editorForm">
            <input type="hidden" name="action" value="save_campaign">
            <input type="hidden" name="id" value="<?php echo (int)$campaign['id']; ?>">

            <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:12px;">
              <div style="display:flex; gap:12px; align-items:center;">
                <button class="btn btn-outline" type="button" id="saveBtnTop">Save</button>
                <button class="btn btn-outline" type="button" id="undoBtn" disabled>Undo</button>
                <button class="btn btn-outline" type="button" id="redoBtn" disabled>Redo</button>
              </div>
              <div>
                <!-- single Review button -->
                <button class="btn btn-primary" type="button" id="reviewTopBtn">Review Details and Send →</button>
              </div>
            </div>

            <div class="editor-shell">
              <!-- LEFT: Modules + Envelope -->
              <div class="editor-left" role="region" aria-label="Design left sidebar">
                <div class="editor-tabs">
                  <div class="tab active" data-tab="build">Build</div>
                  <div class="tab" data-tab="settings">Settings</div>
                  <div class="tab" data-tab="tags">Tags</div>
                  <div class="tab" data-tab="ab">A/B Testing</div>
                </div>

                <div class="modules-pane" id="modulesPane">
                  <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:10px;">
                    <div style="font-weight:600; color:var(--sg-muted);">Add Modules</div>
                    <div style="color:var(--sg-muted); font-size:13px;">Drag or click → drop into the canvas</div>
                  </div>
                  <div class="modules-grid" id="modulesGrid">
                    <div class="module-tile" data-module="image"><div class="icon">🖼️</div>Image</div>
                    <div class="module-tile" data-module="text"><div class="icon">✏️</div>Text</div>
                    <div class="module-tile" data-module="columns"><div class="icon">▦</div>Columns</div>
                    <div class="module-tile" data-module="imgtext"><div class="icon">🧩</div>Image &amp; Text</div>
                    <div class="module-tile" data-module="button"><div class="icon">🔘</div>Button</div>
                    <div class="module-tile" data-module="code"><div class="icon">&lt;&gt;</div>Code</div>
                    <div class="module-tile" data-module="divider"><div class="icon">─</div>Divider</div>
                    <div class="module-tile" data-module="spacer"><div class="icon">↕️</div>Spacer</div>
                    <div class="module-tile" data-module="social"><div class="icon">⚑</div>Social</div>
                    <div class="module-tile" data-module="unsubscribe"><div class="icon">✖</div>Unsubscribe</div>
                  </div>

                </div>

                <div class="envelope">
                  <div style="font-weight:600; margin-bottom:8px;">Envelope</div>
                  <div class="form-group">
                    <label>Subject</label>
                    <input type="text" name="subject" value="<?php echo h($campaign['subject']); ?>" required>
                  </div>
                  <div class="form-group">
                    <label>Preheader</label>
                    <input type="text" name="preheader" value="<?php echo h($campaign['preheader']); ?>">
                  </div>

                  <div class="form-group">
                    <label>
                      From
                      <?php if ($rotOnEditor): ?>
                        <span style="font-weight:400; color:var(--sg-muted);">(taken from each active profile while rotation is ON)</span>
                      <?php endif; ?>
                    </label>
                    <?php if ($rotOnEditor): ?>
                      <select disabled>
                        <option>Rotation enabled — using each profile's From</option>
                      </select>
                    <?php else: ?>
                      <select name="from_email" required>
                        <option value="">Select from Sending Profiles</option>
                        <?php foreach ($profiles as $p): ?>
                          <option value="<?php echo h($p['from_email']); ?>"
                            <?php if ($campaign['from_email'] === $p['from_email']) echo 'selected'; ?>>
                            <?php echo h($p['from_email'].' — '.$p['profile_name']); ?>
                          </option>
                        <?php endforeach; ?>
                      </select>
                    <?php endif; ?>
                  </div>

                  <div class="form-group">
                    <label>Audience (List / segment)</label>
                    <!-- audience now shows contact lists for selection -->
                    <select name="audience">
                      <option value="">Select a contact list or keep free text</option>
                      <?php foreach ($contactLists as $cl):
                        $val = 'list:'.$cl['id'];
                        $selected = ($campaign['audience'] === $val) ? 'selected' : '';
                      ?>
                        <option value="<?php echo h($val); ?>" <?php echo $selected; ?>>
                          <?php echo h($cl['name']).' — '.(int)$cl['contact_count'].' contacts'; ?>
                        </option>
                      <?php endforeach; ?>
                      <option value="custom" <?php if ($campaign['audience'] === 'custom') echo 'selected'; ?>>Custom segment / manual</option>
                    </select>
                    <small class="hint">Choose one of your contact lists. If you need a text/segment, select "Custom" then edit audience in the campaign later.</small>
                  </div>

                  <div class="form-row">
                    <div class="form-group">
                      <label>Sender Name</label>
                      <input type="text" name="sender_name" value="<?php echo h($campaign['sender_name'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                      <label>&nbsp;</label>
                      <div class="checkbox-row">
                        <input type="checkbox" name="unsubscribe_enabled" id="unsubscribe_enabled" <?php if (!empty($campaign['unsubscribe_enabled'])) echo 'checked'; ?>>
                        <label for="unsubscribe_enabled">Enable Unsubscribe Link</label>
                      </div>
                    </div>
                  </div>

                  <div class="form-row">
                    <div class="form-group">
                      <label>&nbsp;</label>
                      <small class="hint">If enabled, an unsubscribe tracking link will be injected into outgoing messages automatically. Recipients who click will be added to the unsubscribes list and blocked from future sends.</small>
                    </div>
                  </div>

                </div>
              </div>

              <!-- RIGHT: Canvas -->
              <div class="editor-right">
                <div class="editor-canvas-top">
                  <div class="canvas-header-left">
                    <div class="subj">Subject: <?php echo $campaign['subject'] ? h($campaign['subject']) : 'Subject'; ?></div>
                    <div class="pre">Preheader: <?php echo $campaign['preheader'] ? h($campaign['preheader']) : ''; ?></div>
                  </div>
                  <div class="canvas-header-right">
                    <button class="btn btn-outline" type="button" id="previewBtn">Preview</button>
                    <button class="btn btn-outline" type="button" id="saveAsTplBtn">Save</button>
                    <!-- single Review button at top -->
                  </div>
                </div>

                <div class="canvas-area" id="canvasArea" data-placeholder="Drag Module Here">
                  <?php
                    // initial canvas content will be hydrated by JS from campaign.html
                    if ($campaign['html']) {
                        echo '<!-- stored html will be loaded into blocks by JS -->';
                    } else {
                        echo '<div class="drag-placeholder">Drag Module Here</div>';
                    }
                  ?>
                </div>

                <div class="canvas-footer">
                  <div style="max-width:780px; margin:0 auto;">
                    <?php if (!empty($campaign['sender_name'])): ?>
                      <?php echo h($campaign['sender_name']); ?>
                    <?php else: ?>
                      <!-- no address lines as requested; show nothing if no sender_name -->
                    <?php endif; ?>
                    <?php if (!empty($campaign['unsubscribe_enabled'])): ?>
                      &nbsp;·&nbsp;<a href="#">Unsubscribe</a>
                    <?php endif; ?>
                  </div>
                </div>
              </div>
            </div>

            <!-- Hidden html textarea (holds campaign html) -->
            <textarea name="html" id="htmlField" style="display:none;"><?php echo h($campaign['html']); ?></textarea>

            <!-- hidden bottom buttons kept for compatibility -->
            <div style="display:none; margin-top:12px; text-align:right;">
              <button type="submit" class="btn btn-outline">Save</button>
<!-- continuing from the snippet above -->
              <button type="submit" name="go_to_review" value="1" class="btn btn-primary">Review Details &amp; Send</button>
            </div>
          </form>

          <!-- HTML Editor Modal (for Code module) -->
          <div class="html-editor-backdrop" id="htmlEditorBackdrop" role="dialog" aria-modal="true">
            <div class="html-editor-panel" role="document">
              <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:8px;">
                <div style="font-weight:600;">HTML Editor</div>
                <button class="btn btn-outline" id="htmlEditorClose">✕</button>
              </div>
              <textarea id="htmlEditorTextarea"><?php echo h($campaign['html']); ?></textarea>
              <div class="html-editor-actions">
                <button class="btn btn-outline" id="htmlEditorCancel">Cancel</button>
                <button class="btn btn-primary" id="htmlEditorSave">Save HTML</button>
              </div>
            </div>
          </div>

          <!-- Include SortableJS for drag & drop -->
          <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>

          <script>
            (function(){
              // Helpers
              function $(sel, root){ return (root || document).querySelector(sel); }
              function $all(sel, root){ return Array.from((root || document).querySelectorAll(sel)); }

              var reviewTop = document.getElementById('reviewTopBtn');
              var form = document.getElementById('editorForm');
              var saveBtnTop = document.getElementById('saveBtnTop');

              // HTML Editor modal elems (declare early so handlers can reference them)
              var htmlBackdrop = document.getElementById('htmlEditorBackdrop');
              var htmlTextarea = document.getElementById('htmlEditorTextarea');
              var htmlClose = document.getElementById('htmlEditorClose');
              var htmlCancel = document.getElementById('htmlEditorCancel');
              var htmlSave = document.getElementById('htmlEditorSave');

              // Toast utility
              function showToast(msg, type){
                var t = document.getElementById('globalToast');
                if(!t) return;
                t.innerText = msg;
                t.className = 'toast show ' + (type === 'error' ? 'error' : 'success');
                setTimeout(function(){
                  t.className = 'toast';
                }, 2000);
              }

              // Save behavior (top save) - use AJAX to enable autosave UX
              function doSaveAjax(callback){
                syncCanvasToHtmlField();
                var fd = new FormData(form);
                // Ensure action=save_campaign present
                fd.set('action', 'save_campaign');

                fetch(window.location.pathname + window.location.search, {
                  method: 'POST',
                  body: fd,
                  credentials: 'same-origin',
                  headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                  }
                }).then(function(resp){
                  if (!resp.ok) throw new Error('Save failed');
                  return resp.json().catch(function(){ return {ok:1}; });
                }).then(function(json){
                  showToast('Saved', 'success');
                  if (typeof callback === 'function') callback(null, json);
                }).catch(function(err){
                  showToast('Save error', 'error');
                  // fallback to normal submit if desired
                  if (confirm('Autosave failed, submit full form instead?')) {
                    form.submit();
                  }
                  if (typeof callback === 'function') callback(err);
                });
              }

              function doSave() {
                doSaveAjax();
              }
              if (saveBtnTop) saveBtnTop.addEventListener('click', doSave);

              // Autosave every 7 seconds (only on editor page)
              var autosaveTimer = setInterval(function(){
                // only run autosave if form exists and page is visible
                if (document.visibilityState === 'visible') {
                  doSaveAjax();
                }
              }, 7000);

              // When user clicks Review: if HTML modal is open, automatically save modal contents first
              function doReviewSubmit(){
                // if modal open, persist modal content programmatically (avoid invoking handlers via click())
                if (htmlBackdrop && htmlBackdrop.classList.contains('show')) {
                  try {
                    // Persist raw HTML into hidden field
                    if (typeof htmlField !== 'undefined' && htmlField !== null) {
                      htmlField.value = htmlTextarea.value;
                    } else {
                      // htmlField may be defined later; fallback to finding it now
                      var hf = document.getElementById('htmlField');
                      if (hf) hf.value = htmlTextarea.value;
                    }
                    // Rehydrate canvas from new HTML so syncCanvasToHtmlField picks latest content
                    if (typeof loadHtmlIntoCanvas === 'function') {
                      loadHtmlIntoCanvas(htmlTextarea.value);
                    }
                    // Close modal visually
                    if (typeof closeHtmlEditor === 'function') {
                      closeHtmlEditor();
                    } else {
                      htmlBackdrop.classList.remove('show');
                      htmlBackdrop.style.display = 'none';
                    }
                  } catch (e) {
                    // if anything goes wrong, gracefully continue to submit whatever we have
                  }

                  var hidden = document.createElement('input');
                  hidden.type = 'hidden';
                  hidden.name = 'go_to_review';
                  hidden.value = '1';
                  form.appendChild(hidden);
                  // ensure latest HTML is synced
                  syncCanvasToHtmlField();
                  form.submit();
                  return;
                }
                // otherwise normal submit
                var hidden = document.createElement('input');
                hidden.type = 'hidden';
                hidden.name = 'go_to_review';
                hidden.value = '1';
                form.appendChild(hidden);
                // ensure html saved from canvas
                syncCanvasToHtmlField();
                form.submit();
              }
              if (reviewTop) reviewTop.addEventListener('click', doReviewSubmit);

              // small tab UI (non-functional placeholders)
              var tabs = document.querySelectorAll('.editor-left .tab');
              tabs.forEach(function(t){
                t.addEventListener('click', function(){
                  tabs.forEach(function(x){ x.classList.remove('active'); });
                  t.classList.add('active');
                });
              });

              var modulesGrid = document.getElementById('modulesGrid');
              var modulesPane = document.getElementById('modulesPane');
              var canvas = document.getElementById('canvasArea');
              var htmlField = document.getElementById('htmlField');

              function openHtmlEditor() {
                htmlTextarea.value = htmlField.value || '';
                htmlBackdrop.classList.add('show');
                htmlBackdrop.style.display = 'flex';
                htmlTextarea.focus();
              }
              function closeHtmlEditor() {
                htmlBackdrop.classList.remove('show');
                htmlBackdrop.style.display = 'none';
              }

              // Build block markup for modules
              function createBlockHtmlByModule(module) {
                    switch(module) {
                      case 'image':
                        return '<div style="max-width:100%;text-align:center;"><img src="https://via.placeholder.com/600x200" alt="Image" style="max-width:100%;"></div>';
                      case 'text':
                        return '<div style="max-width:100%;font-size:15px;line-height:1.5;color:#333;"><p>Edit text by opening the Code module or double-click this block (use Code for advanced changes).</p></div>';
                      case 'button':
                        return '<div style="text-align:center;margin:16px;"><a href="#" style="background:var(--sg-blue);color:#fff;padding:10px;"></div>';
                                                case 'columns':
                        return '<div style="display:flex;gap:10px;"><div style="flex:1;background:#fafafa;padding:10px;">Column 1</div><div style="flex:1;background:#fafafa;padding:10px;">Column 2</div></div>';
                      case 'imgtext':
                        return '<div style="display:flex;gap:12px;align-items:center;"><img src="https://via.placeholder.com/160x100" style="width:160px;height:auto;"><div>Text next to image</div></div>';
                      case 'code':
                        // Removed the static "Add Code" placeholder from the default code module body
                        return '<div class="code-module"><div class="code-module-header"><span></span><span class="tools"><button type="button" data-tool="edit" title="Edit code" style="background:transparent;border:none;color:#fff;cursor:pointer;">&lt;&gt;</button><button type="button" data-tool="delete" title="Delete" style="background:transparent;border:none;color:#fff;cursor:pointer;">🗑</button></span></div><div class="code-module-body"></div></div>';
                      default:
                        return '<div>Module: '+module+'</div>';
                    }
                  }

                  // Create canvas block element (wrapper with remove button)
                  // Accepts contentHtml or a module wrapper (like code-module)
                  function makeCanvasBlock(contentHtml) {
                    var wrapper = document.createElement('div');
                    wrapper.className = 'canvas-block';
                    var removeBtn = document.createElement('div');
                    removeBtn.className = 'block-remove';
                    removeBtn.title = 'Remove block';
                    removeBtn.innerHTML = '✕';
                    removeBtn.addEventListener('click', function(){
                      // confirm removal
                      if (confirm('Remove this block?')) {
                        wrapper.remove();
                        updatePlaceholderVisibility();
                      }
                    });

                    var content = document.createElement('div');
                    content.className = 'block-content';
                    content.innerHTML = contentHtml || '';

                    // If the block contains a code-module, wire up header tools and editing behavior
                    var codeModuleEl = content.querySelector('.code-module');
                    if (codeModuleEl) {
                      // mark dataset for later use
                      wrapper.dataset.module = 'code';

                      var header = codeModuleEl.querySelector('.code-module-header');
                      var body = codeModuleEl.querySelector('.code-module-body');

                      // tool handlers
                      header.addEventListener('click', function(e){
                        // if clicked on tool buttons handle separately
                        var t = e.target;
                        if (t && t.getAttribute && t.getAttribute('data-tool') === 'delete') {
                          if (confirm('Delete this code block?')) {
                            wrapper.remove();
                            updatePlaceholderVisibility();
                          }
                          return;
                        }
                        // open the block-level HTML editor for this code body
                        openBlockEditor(body);
                      });

                      // support double-click edit for code body as well
                      body.addEventListener('dblclick', function(){
                        openBlockEditor(body);
                      });
                    } else {
                      // double-click to edit via HTML editor (for general blocks)
                      content.addEventListener('dblclick', function(){
                        // open code editor with this block's HTML only
                        htmlTextarea.value = content.innerHTML;
                        htmlBackdrop.classList.add('show');
                        htmlBackdrop.style.display = 'flex';
                        var oneOff = function(e){
                          e.preventDefault();
                          content.innerHTML = htmlTextarea.value;
                          htmlBackdrop.classList.remove('show');
                          htmlBackdrop.style.display = 'none';
                          htmlSave.removeEventListener('click', oneOff);
                          if (defaultSaveHandler) {
                            htmlSave.addEventListener('click', defaultSaveHandler);
                          }
                          syncCanvasToHtmlField();
                        };
                        // replace
                        // replace handlers for this one-off edit
                        if (defaultSaveHandler) htmlSave.removeEventListener('click', defaultSaveHandler);
                        htmlSave.addEventListener('click', oneOff);
                      });
                    }

                    wrapper.appendChild(removeBtn);
                    wrapper.appendChild(content);
                    return wrapper;
                  }

                  // Open block-level HTML editor (edits only the body of a code module or a block container)
                  function openBlockEditor(bodyElement) {
                    // set the textarea to the current innerHTML of the block body
                    htmlTextarea.value = bodyElement.innerHTML;
                    htmlBackdrop.classList.add('show');
                    htmlBackdrop.style.display = 'flex';
                    htmlTextarea.focus();

                    // assign a one-off save handler that updates this bodyElement
                    var oneOff = function(e){
                      e.preventDefault();
                      bodyElement.innerHTML = htmlTextarea.value;
                      htmlBackdrop.classList.remove('show');
                      htmlBackdrop.style.display = 'none';
                      // restore global handler
                      htmlSave.removeEventListener('click', oneOff);
                      if (defaultSaveHandler) {
                        htmlSave.addEventListener('click', defaultSaveHandler);
                      }
                      syncCanvasToHtmlField();
                    };

                    // ensure previous onclick is cleared and attach oneOff
                    if (defaultSaveHandler) htmlSave.removeEventListener('click', defaultSaveHandler);
                    htmlSave.addEventListener('click', oneOff);
                  }

                  // Sync: read all canvas blocks and set htmlField
                  function syncCanvasToHtmlField() {
                    // take innerHTML of all .block-content inside canvas, concatenate
                    var blocks = canvas.querySelectorAll('.canvas-block .block-content');
                    if (blocks.length === 0) {
                      htmlField.value = '';
                      return;
                    }
                    var html = '';
                    blocks.forEach(function(b){
                      // Check if this block contains a code-module
                      var codeModule = b.querySelector('.code-module');
                      if (codeModule) {
                        // For code modules, extract only the body content (skip the header with emoji/buttons)
                        var codeBody = codeModule.querySelector('.code-module-body');
                        if (codeBody) {
                          html += codeBody.innerHTML;
                        } else {
                          // Fallback: if body not found, use full block content to avoid data loss
                          html += b.innerHTML;
                        }
                      } else {
                        // For non-code modules, include the entire block content
                        html += b.innerHTML;
                      }
                    });
                    htmlField.value = html;
                  }

                  // Update placeholder if empty
                  function updatePlaceholderVisibility() {
                    var hasBlocks = canvas.querySelectorAll('.canvas-block').length > 0;
                    var placeholder = canvas.querySelector('.drag-placeholder');
                    if (!hasBlocks) {
                      if (!placeholder) {
                        var ph = document.createElement('div');
                        ph.className = 'drag-placeholder';
                        ph.innerText = 'Drag Module Here';
                        canvas.appendChild(ph);
                      }
                    } else {
                      if (placeholder) placeholder.remove();
                    }
                  }

                  // Default html save handler for modal (saves entire canvas)
                  function defaultHtmlSaveHandler(e){
                    e.preventDefault();
                    // set hidden field and update canvas preview
                    htmlField.value = htmlTextarea.value;
                    // rehydrate canvas blocks from new html (single big block)
                    loadHtmlIntoCanvas(htmlTextarea.value);
                    closeHtmlEditor();
                  }
                  var defaultSaveHandler = defaultHtmlSaveHandler;
                  if (htmlSave) htmlSave.addEventListener('click', defaultSaveHandler);

                  if (htmlClose) htmlClose.addEventListener('click', function(){ closeHtmlEditor(); });
                  if (htmlCancel) htmlCancel.addEventListener('click', function(e){ e.preventDefault(); closeHtmlEditor(); });

                  // Parse existing htmlField value into canvas blocks (split top-level nodes)
                  function loadHtmlIntoCanvas(rawHtml) {
                    canvas.innerHTML = ''; // clear
                    if (!rawHtml || rawHtml.trim() === '') {
                      updatePlaceholderVisibility();
                      return;
                    }
                    var tmp = document.createElement('div');
                    tmp.innerHTML = rawHtml;
                    // if there are multiple top-level nodes, create a block for each
                    var children = Array.from(tmp.childNodes).filter(function(n){
                      // ignore empty text nodes
                      return !(n.nodeType === 3 && !n.textContent.trim());
                    });
                    if (children.length === 0) {
                      updatePlaceholderVisibility();
                      return;
                    }
                    children.forEach(function(node){
                      var html = (node.outerHTML !== undefined) ? node.outerHTML : node.textContent;
                      var blockEl = makeCanvasBlock(html);
                      canvas.appendChild(blockEl);
                    });
                    updatePlaceholderVisibility();
                  }

                  // Initialize Sortable for modules (pull: clone)
                  var modulesSortable = Sortable.create(document.getElementById('modulesGrid'), {
                    group: { name: 'modules', pull: 'clone', put: false },
                    sort: false,
                    animation: 150,
                    onEnd: function (evt) {
                      // modules grid end — clicks are handled separately
                    }
                  });

                  // Initialize Sortable on canvas to accept modules and sort existing blocks
                  var canvasSortable = Sortable.create(canvas, {
                    group: { name: 'modules', pull: false, put: true },
                    animation: 150,
                    ghostClass: 'sortable-ghost',
                    onAdd: function (evt) {
                      // when an item is dragged from modules into canvas, evt.item is the clone element (module-tile)
                      var dragged = evt.item;
                      var module = dragged.getAttribute('data-module');
                      // create a real block with contentHtml
                      var contentHtml = createBlockHtmlByModule(module);
                      var blockEl = makeCanvasBlock(contentHtml);
                      // replace the dragged placeholder with the real block
                      dragged.parentNode.replaceChild(blockEl, dragged);
                      syncCanvasToHtmlField();

                      // If the inserted module is code, open the block-level editor immediately
                      if (module === 'code') {
                        // find the block's body node and open block editor
                        var body = blockEl.querySelector('.code-module-body');
                        if (body) {
                          // small delay to ensure DOM is ready
                          setTimeout(function(){ openBlockEditor(body); }, 50);
                        }
                      }
                    },
                    onUpdate: function(evt){
                      // reorder happened
                      syncCanvasToHtmlField();
                    },
                    onRemove: function(evt){
                      syncCanvasToHtmlField();
                    }
                  });

                  // Also support clicking module tiles to insert at end
                  $all('.module-tile').forEach(function(t){
                    t.addEventListener('click', function(){
                      var module = t.getAttribute('data-module');
                      if (module === 'code') {
                        // open full HTML editor (editing whole message)
                        openHtmlEditor();
                        // defaultSaveHandler already handles full save
                        return;
                      }
                      var html = createBlockHtmlByModule(module);
                      var block = makeCanvasBlock(html);
                      // if placeholder exists, replace it
                      var placeholder = canvas.querySelector('.drag-placeholder');
                      if (placeholder) {
                        placeholder.remove();
                      }
                      canvas.appendChild(block);
                      // ensure block is sortable (Sortable automatically includes it)
                      syncCanvasToHtmlField();
                    });
                  });

                  // Hydrate initial html into canvas
                  loadHtmlIntoCanvas(htmlField.value);

                  // If no HTML and we want a code module shown by default, create it and open editor
                  // (This matches the requested UX: when first inserting code or when campaign empty show code editor option)
                  if ((!htmlField.value || htmlField.value.trim() === '') && canvas.querySelectorAll('.canvas-block').length === 0) {
                    // leave placeholder visible; user will insert modules intentionally
                  }

                  // Keep canvas and htmlField in sync on form submit
                  form.addEventListener('submit', function(){
                    syncCanvasToHtmlField();
                  });

                  // Utility: preview and save as template placeholders (not implemented)
                  var previewBtn = document.getElementById('previewBtn');
                  var saveAsTplBtn = document.getElementById('saveAsTplBtn');
                  if (previewBtn) previewBtn.addEventListener('click', function(){ alert('Preview not implemented in this demo.'); });
                  if (saveAsTplBtn) saveAsTplBtn.addEventListener('click', function(){ alert('Save as template not implemented.'); });

                  // ESC to close html modal
                  document.addEventListener('keydown', function(e){
                    if (e.key === 'Escape') {
                      if (htmlBackdrop && htmlBackdrop.classList.contains('show')) closeHtmlEditor();
                    }
                  });

                  // Accessibility: when dragging from modules, set aria-grabbed (basic)
                                    $all('.module-tile').forEach(function(t){
                    t.setAttribute('draggable', 'true');
                  });

                })();
              </script>

              <!-- Global toast element (used by autosave / success / error messages) -->
              <div id="globalToast" class="toast" role="status" aria-live="polite" aria-atomic="true"></div>

            <?php } ?>

          <?php elseif ($page === 'review'): ?>
            <?php
              $id = (int)($_GET['id'] ?? 0);
              $campaign = get_campaign($pdo, $id);
              if (!$campaign) {
                echo "<p>Campaign not found.</p>";
              } else {
                // Prepare preview HTML (raw)
                $previewHtml = $campaign['html'] ?: '<div style="padding:24px;color:#6B778C;">(No content yet — add modules in the Editor)</div>';
                if (!empty($campaign['unsubscribe_enabled'])) {
                    $basePreviewUrl = get_base_url() . '?t=unsubscribe&cid=' . (int)$campaign['id'];
                    $previewHtml .= '<div style="text-align:center;margin-top:20px;color:#1F2933;font-size:13px;">';
                    $previewHtml .= '<a href="' . $basePreviewUrl . '" style="color:#1A82E2;margin-right:8px;">Unsubscribe</a>';
                    $previewHtml .= '<span style="color:#6B778C;">-</span>';
                    $previewHtml .= '<a href="' . $basePreviewUrl . '" style="color:#1A82E2;margin-left:8px;">Unsubscribe Preferences</a>';
                    $previewHtml .= '</div>';
                }
                $testSent = isset($_GET['test_sent']);
                $sentFlag = isset($_GET['sent']);
                $sendOk = isset($_GET['send_ok']);
                $sendErr = isset($_GET['send_err']);
            ?>
              <div class="page-title">Review &amp; Send — <?php echo h($campaign['name']); ?></div>
              <div class="page-subtitle">Confirm your envelope settings and preview the content before sending.</div>

              <div class="review-grid">
                <div>
                  <div class="review-summary">
                    <form method="post" id="sendForm">
                      <input type="hidden" name="action" value="send_campaign">
                      <input type="hidden" name="id" value="<?php echo (int)$campaign['id']; ?>">

                      <div class="form-group">
                        <label>Single Send Name</label>
                        <input type="text" name="name" value="<?php echo h($campaign['name']); ?>" disabled>
                      </div>

                      <div class="form-group">
                        <label>From Sender</label>
                        <select name="from_email">
                          <option value=""><?php echo $campaign['from_email'] ? h($campaign['from_email']) : 'Select sender'; ?></option>
                          <?php foreach ($profiles as $p): ?>
                            <option value="<?php echo h($p['from_email']); ?>" <?php if ($campaign['from_email'] === $p['from_email']) echo 'selected'; ?>>
                              <?php echo h($p['from_email'] . ' — ' . $p['profile_name']); ?>
                            </option>
                          <?php endforeach; ?>
                        </select>
                      </div>

                      <div class="form-group">
                        <label>Subject</label>
                        <input type="text" name="subject" value="<?php echo h($campaign['subject']); ?>">
                      </div>

                      <div class="form-group">
                        <label>Preheader</label>
                        <input type="text" name="preheader" value="<?php echo h($campaign['preheader']); ?>">
                      </div>

                      <div class="form-group">
                        <label>Unsubscribe Group</label>
                        <div style="display:flex; align-items:center; gap:10px;">
                          <select name="unsubscribe_group" disabled>
                            <option value="">Default</option>
                          </select>
                          <div class="checkbox-row" style="margin:0;">
                            <input type="checkbox" id="rv_unsub" name="unsubscribe_enabled" <?php if (!empty($campaign['unsubscribe_enabled'])) echo 'checked'; ?>>
                            <label for="rv_unsub" style="font-weight:600;margin:0;">Enable Unsubscribe</label>
                          </div>
                        </div>
                      </div>

                      <div class="form-group">
                        <label>Schedule</label>
                        <input type="text" value="Send Immediately" disabled>
                      </div>

                      <div class="form-group">
                        <label>Send To Recipients</label>
                        <select name="audience_select" id="audienceSelect" style="width:100%; padding:8px;">
                          <option value="">Select recipients</option>
                          <?php foreach ($contactLists as $cl):
                            $val = 'list:'.$cl['id'];
                          ?>
                            <option value="<?php echo h($val); ?>"><?php echo h($cl['name']).' — '.(int)$cl['contact_count'].' contacts'; ?></option>
                          <?php endforeach; ?>
                          <option value="manual">Manual - Enter emails</option>
                        </select>
                      </div>

                      <div id="manualRecipientsWrap" style="display:none; margin-top:10px;">
                        <label style="font-weight:600; display:block; margin-bottom:6px;">Manual recipients (one per line)</label>
                        <textarea name="test_recipients" placeholder="test1@example.com&#10;test2@example.com" rows="6" style="width:100%;"></textarea>
                      </div>

                      <div style="display:flex; justify-content:flex-end; gap:8px; margin-top:12px;">
                        <a href="?page=editor&id=<?php echo (int)$campaign['id']; ?>" class="btn btn-outline">Back to Editor</a>
                        <button type="submit" class="btn btn-primary">Send Immediately</button>
                      </div>
                    </form>

                    <div style="margin-top:18px;">
                      <div style="font-weight:600; margin-bottom:8px;">Test Your Email</div>
                      <div style="color:var(--sg-muted); margin-bottom:8px;">Test your email before sending to your recipients.</div>

                      <form method="post" style="margin-top:12px;" id="testForm">
                        <input type="hidden" name="action" value="send_test_message">
                        <input type="hidden" name="id" value="<?php echo (int)$campaign['id']; ?>">
                        <div>
                          <label style="display:block; font-weight:600; margin-bottom:6px;">Send a Test Email</label>
                          <input type="text" name="test_addresses" placeholder="Enter up to 10 email addresses, separated by commas" style="width:100%; padding:10px;">
                        </div>
                        <div style="display:flex; justify-content:space-between; align-items:center; margin-top:10px;">
                          <div style="color:var(--sg-muted); font-size:12px;">You can send test messages to up to 10 addresses.</div>
                          <button class="btn btn-primary" type="submit">Send Test Message</button>
                        </div>
                      </form>

                      <small class="hint">Test messages are sent immediately and recorded as test events (they won't change campaign status).</small>
                    </div>

                  </div>
                </div>

                <div>
                  <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:12px;">
                    <div style="font-weight:600;">Content preview</div>
                    <div style="color:var(--sg-muted); font-size:12px;">Desktop | Mobile | Plain Text</div>
                  </div>
                  <div class="preview-box" id="previewBox">
                    <?php echo $previewHtml; ?>
                  </div>
                </div>
              </div>

              <script>
                (function(){
                  var aud = document.getElementById('audienceSelect');
                  var manualWrap = document.getElementById('manualRecipientsWrap');

                  function updateManualVisibility() {
                    if (aud && manualWrap) {
                      if (aud.value === 'manual') {
                        manualWrap.style.display = '';
                      } else {
                        manualWrap.style.display = 'none';
                      }
                    }
                  }
                  if (aud) {
                    aud.addEventListener('change', updateManualVisibility);
                    updateManualVisibility();
                  }

                  var rvUnsub = document.getElementById('rv_unsub');
                  var previewBox = document.getElementById('previewBox');
                  if (rvUnsub && previewBox) {
                    rvUnsub.addEventListener('change', function(){
                      var existing = previewBox.querySelector('.preview-unsubscribe-block');
                      if (rvUnsub.checked) {
                        if (!existing) {
                          var div = document.createElement('div');
                          div.className = 'preview-unsubscribe-block';
                          div.style.textAlign = 'center';
                          div.style.marginTop = '20px';
                          div.innerHTML = '<a href="<?php echo get_base_url() . '?t=unsubscribe&cid=' . (int)$campaign['id']; ?>" style="color:#1A82E2;margin-right:8px;">Unsubscribe</a> - <a href="<?php echo get_base_url() . '?t=unsubscribe&cid=' . (int)$campaign['id']; ?>" style="color:#1A82E2;margin-left:8px;">Unsubscribe Preferences</a>';
                          previewBox.appendChild(div);
                        }
                      } else {
                        if (existing) existing.remove();
                      }
                    });
                  }

                  // Show toast if test was sent or campaign sent flag present
                  <?php if ($testSent): ?>
                    document.addEventListener('DOMContentLoaded', function(){ showToast('Test message sent', 'success'); });
                  <?php endif; ?>
                })();
              </script>

            <?php } ?>

          <?php elseif ($page === 'stats'): ?>
            <?php
              $id = (int)($_GET['id'] ?? 0);
              $campaign = get_campaign($pdo, $id);
              if (!$campaign) {
                echo "<p>Campaign not found.</p>";
              } else {
                $stats = get_campaign_stats($pdo, $id);
                $delivered = max(1, (int)$stats['delivered']);
                $openRate  = $delivered ? round(($stats['open'] / $delivered) * 100, 1) : 0;
                $clickRate = $delivered ? round(($stats['click'] / $delivered) * 100, 1) : 0;

                // Per-profile aggregation for this campaign (PHP-side)
                $profilesAll = get_profiles($pdo);
                $profilesMap = [];
                foreach ($profilesAll as $p) $profilesMap[(int)$p['id']] = $p;

                $perProfile = [];
                try {
                    $stmtp = $pdo->prepare("SELECT event_type, details FROM events WHERE campaign_id = ?");
                    $stmtp->execute([$id]);
                    foreach ($stmtp as $row) {
                        $etype = $row['event_type'];
                        $details = json_decode($row['details'], true);
                        if (!is_array($details)) continue;
                        $pid = isset($details['profile_id']) ? (int)$details['profile_id'] : 0;
                        if ($pid <= 0) continue;
                        if (!isset($perProfile[$pid])) $perProfile[$pid] = ['sent'=>0,'delivered'=>0,'bounces'=>0,'unsubscribes'=>0,'clicks'=>0,'opens'=>0];
                        // Count sent attempts: delivered + bounce + deferred
                        if (in_array($etype, ['delivered','bounce','deferred'])) $perProfile[$pid]['sent']++;
                        if ($etype === 'delivered') $perProfile[$pid]['delivered']++;
                        if ($etype === 'bounce') $perProfile[$pid]['bounces']++;
                        if ($etype === 'unsubscribe' || $etype === 'skipped_unsubscribe') $perProfile[$pid]['unsubscribes']++;
                        if ($etype === 'click') $perProfile[$pid]['clicks']++;
                        if ($etype === 'open') $perProfile[$pid]['opens']++;
                    }
                } catch (Exception $e) {}
            ?>
              <div class="page-title">Stats — <?php echo h($campaign['name']); ?></div>
              <div class="page-subtitle">Delivery &amp; engagement metrics for your Single Send.</div>

              <div class="card" style="margin-bottom:18px;">
                <div class="card-header">
                  <div>
                    <div class="card-title">Summary</div>
                  </div>
                  <a href="?page=list" class="btn btn-outline">← Back to Single Sends</a>
                </div>
                <div class="form-row">
                  <div class="form-group">
                    <label>Subject</label>
                    <div><?php echo h($campaign['subject']); ?></div>
                  </div>
                  <div class="form-group">
                    <label>Sent at</label>
                    <div><?php echo $campaign['sent_at'] ? h($campaign['sent_at']) : '<span class="hint">Not sent</span>'; ?></div>
                  </div>
                </div>
              </div>

              <!-- Real-time Campaign Progress -->
              <?php 
                $progress = get_campaign_progress($pdo, $id);
                if ($progress['status'] === 'queued' || $progress['status'] === 'sending' || ($campaign['status'] === 'sending' && $progress['total'] > 0)):
              ?>
              <div class="card" style="margin-bottom:18px;" id="progressCard">
                <div class="card-header">
                  <div>
                    <div class="card-title">Campaign Progress</div>
                    <div class="card-subtitle">Real-time sending progress (updates automatically)</div>
                  </div>
                </div>
                <div style="padding:16px;">
                  <div style="margin-bottom:12px; font-size:18px; font-weight:600;">
                    <span id="progressText"><?php echo $progress['sent']; ?> / <?php echo $progress['total']; ?> sent (<?php echo $progress['percentage']; ?>%)</span>
                  </div>
                  <div style="background:#E4E7EB; border-radius:4px; height:24px; overflow:hidden; position:relative;">
                    <div id="progressBar" style="background:linear-gradient(90deg, #1A82E2, #3B9EF3); height:100%; width:<?php echo $progress['percentage']; ?>%; transition:width 0.3s;"></div>
                  </div>
                  <div style="margin-top:8px; color:#6B778C; font-size:13px;" id="progressStatus">Status: <?php echo ucfirst($progress['status']); ?></div>
                </div>
              </div>
              <?php endif; ?>

              <!-- Simplified header stats matching requested layout -->
              <div class="header-stats">
                <div class="stat-item">
                  <div class="stat-num"><?php echo (int)$stats['target']; ?></div>
                  <div class="stat-label">Emails Triggered</div>
                </div>
                <div class="stat-item">
                  <div class="stat-num"><?php echo (int)$stats['delivered']; ?></div>
                  <div class="stat-label">Delivered</div>
                </div>
                <div class="stat-item">
                  <div class="stat-num"><?php echo (int)$stats['open']; ?></div>
                  <div class="stat-label">Unique Opens</div>
                </div>
                <div class="stat-item">
                  <div class="stat-num"><?php echo (int)$stats['click']; ?></div>
                  <div class="stat-label">Unique Clicks</div>
                </div>
                <div class="stat-item">
                  <div class="stat-num"><?php echo (int)$stats['bounce']; ?></div>
                  <div class="stat-label">Bounces</div>
                </div>
                <div class="stat-item">
                  <div class="stat-num"><?php echo (int)$stats['unsubscribe']; ?></div>
                  <div class="stat-label">Unsubscribes</div>
                </div>
                <div class="stat-item">
                  <div class="stat-num"><?php echo (int)$stats['spam']; ?></div>
                  <div class="stat-label">Spam Reports</div>
                </div>
              </div>

              <div style="margin-bottom:18px;">
                <!-- Placeholder for chart (not implemented) -->
                <div style="background:#fff;border:1px solid var(--sg-border);border-radius:6px;height:320px;"></div>
              </div>

              <div class="card" style="margin-bottom:18px;">
                <div class="card-header">
                  <div>
                    <div class="card-title">By Sending Profile (this campaign)</div>
                    <div class="card-subtitle">Counts of sends, delivered, bounces and unsubscribes per profile for this campaign.</div>
                  </div>
                </div>
                <table class="table">
                  <thead>
                    <tr>
                      <th>Profile</th>
                      <th>From Email</th>
                      <th>Sent Attempts</th>
                      <th>Delivered</th>
                      <th>Bounces</th>
                      <th>Unsubscribes</th>
                      <th>Opens</th>
                      <th>Clicks</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php if (empty($perProfile)): ?>
                      <tr>
                        <td colspan="8" style="text-align:center; padding:16px; color:var(--sg-muted);">No per-profile events found for this campaign.</td>
                      </tr>
                    <?php else: ?>
                      <?php foreach ($perProfile as $pid => $vals):
                        $pf = $profilesMap[$pid] ?? ['profile_name'=>'Profile #'.$pid,'from_email'=>''];
                      ?>
                        <tr>
                          <td><?php echo h($pf['profile_name']); ?></td>
                          <td><?php echo h($pf['from_email']); ?></td>
                          <td><?php echo (int)$vals['sent']; ?></td>
                          <td><?php echo (int)$vals['delivered']; ?></td>
                          <td><?php echo (int)$vals['bounces']; ?></td>
                          <td><?php echo (int)$vals['unsubscribes']; ?></td>
                          <td><?php echo (int)$vals['opens']; ?></td>
                          <td><?php echo (int)$vals['clicks']; ?></td>
                        </tr>
                      <?php endforeach; ?>
                    <?php endif; ?>
                  </tbody>
                </table>
              </div>

              <div class="card">
                <div class="card-header">
                  <div>
                    <div class="card-title">Raw Events</div>
                    <div class="card-subtitle">Last 50 events (demo).</div>
                  </div>
                </div>
                <table class="table">
                  <thead>
                    <tr>
                      <th>Time</th>
                      <th>Type</th>
                      <th>Details</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php
                      $stmt = $pdo->prepare("SELECT * FROM events WHERE campaign_id=? ORDER BY created_at DESC LIMIT 50");
                      $stmt->execute([$id]);
                      $evs = $stmt->fetchAll(PDO::FETCH_ASSOC);
                      if (empty($evs)):
                    ?>
                      <tr>
                        <td colspan="3" style="text-align:center; padding:16px; color:var(--sg-muted);">
                          No events yet.
                        </td>
                      </tr>
                    <?php else: ?>
                      <?php foreach ($evs as $ev):
                        $dt = h($ev['created_at']);
                        $type = strtoupper($ev['event_type']);
                        $details = $ev['details'] ? h($ev['details']) : '—';
                      ?>
                        <tr>
                          <td><?php echo $dt; ?></td>
                          <td><?php echo $type; ?></td>
                          <td><code style="font-size:11px;"><?php echo $details; ?></code></td>
                        </tr>
                      <?php endforeach; ?>
                    <?php endif; ?>
                  </tbody>
                </table>
              </div>

              <!-- Real-time Progress JavaScript -->
              <script>
                (function(){
                  var campaignId = <?php echo (int)$id; ?>;
                  var progressCard = document.getElementById('progressCard');
                  var progressBar = document.getElementById('progressBar');
                  var progressText = document.getElementById('progressText');
                  var progressStatus = document.getElementById('progressStatus');
                  
                  if (progressCard) {
                    // Poll progress API every 2 seconds
                    var pollInterval = setInterval(function(){
                      fetch('?api=progress&campaign_id=' + campaignId)
                        .then(function(r){ return r.json(); })
                        .then(function(data){
                          if (data.success && data.progress) {
                            var prog = data.progress;
                            var stats = data.stats || {};
                            
                            // Update progress bar
                            if (progressBar) {
                              progressBar.style.width = prog.percentage + '%';
                            }
                            
                            // Update progress text
                            if (progressText) {
                              progressText.textContent = prog.sent + ' / ' + prog.total + ' sent (' + prog.percentage + '%)';
                            }
                            
                            // Update status
                            if (progressStatus) {
                              progressStatus.textContent = 'Status: ' + prog.status.charAt(0).toUpperCase() + prog.status.slice(1);
                            }
                            
                            // Update stats in header
                            var statItems = document.querySelectorAll('.stat-item .stat-num');
                            if (statItems.length >= 7) {
                              // statItems[0] is Emails Triggered (don't update)
                              if (statItems[1]) statItems[1].textContent = stats.delivered || 0;
                              if (statItems[2]) statItems[2].textContent = stats.open || 0;
                              if (statItems[3]) statItems[3].textContent = stats.click || 0;
                              if (statItems[4]) statItems[4].textContent = stats.bounce || 0;
                              if (statItems[5]) statItems[5].textContent = stats.unsubscribe || 0;
                              // statItems[6] is Spam Reports (don't update from this API)
                            }
                            
                            // Stop polling if completed
                            if (prog.status === 'completed' || prog.status === 'sent') {
                              clearInterval(pollInterval);
                              // Hide progress card after 3 seconds
                              setTimeout(function(){
                                if (progressCard) {
                                  progressCard.style.transition = 'opacity 0.5s';
                                  progressCard.style.opacity = '0';
                                  setTimeout(function(){
                                    progressCard.style.display = 'none';
                                  }, 500);
                                }
                              }, 3000);
                            }
                          }
                        })
                        .catch(function(err){
                          console.error('Progress poll error:', err);
                        });
                    }, 2000);
                  }
                })();
              </script>

            <?php } ?>

          <?php elseif ($page === 'contacts'): ?>
            <?php
              $listId   = isset($_GET['list_id']) ? (int)$_GET['list_id'] : 0;
              $showCreate = isset($_GET['show_create']);
            ?>

            <?php if ($listId === 0): ?>
              <?php
                $lists = get_contact_lists($pdo);
              ?>
              <div class="page-title">Contact Lists</div>
              <div class="page-subtitle">Manage lists and segments used by your Single Sends, like SendGrid Contacts.</div>

              <div class="card">
                <div class="card-header">
                  <div>
                    <div class="card-title">Lists</div>
                    <div class="card-subtitle">Global list and individual marketing lists.</div>
                  </div>
                  <div style="display:flex; gap:8px;">
                    <a href="?page=contacts&show_create=1" class="btn btn-outline">Create</a>
                    <?php if (!empty($lists)): ?>
                      <a href="?page=contacts&list_id=<?php echo (int)$lists[0]['id']; ?>" class="btn btn-primary">Add Contacts</a>
                    <?php endif; ?>
                  </div>
                </div>

                                <?php if (empty($lists)): ?>
                  <div class="card" style="margin-top:12px;">
                    <div class="hint">No contact lists found. Create one to start adding contacts.</div>
                  </div>
                <?php else: ?>
                  <table class="table">
                    <thead>
                      <tr>
                        <th>List Name</th>
                        <th>Type</th>
                        <th>Contacts</th>
                        <th>Created</th>
                        <th>Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php foreach ($lists as $l): ?>
                        <tr>
                          <td><a href="?page=contacts&list_id=<?php echo (int)$l['id']; ?>"><?php echo h($l['name']); ?></a></td>
                          <td><?php echo h($l['type']); ?></td>
                          <td><?php echo (int)$l['contact_count']; ?></td>
                          <td><?php echo h($l['created_at']); ?></td>
                          <td>
                            <a class="btn-mini" href="?page=contacts&list_id=<?php echo (int)$l['id']; ?>">View</a>
                            <form method="post" style="display:inline;">
                              <input type="hidden" name="action" value="delete_contact_list">
                              <input type="hidden" name="list_id" value="<?php echo (int)$l['id']; ?>">
                              <button type="submit" class="btn-mini" onclick="return confirm('Delete this list and its contacts?');">Delete</button>
                            </form>
                          </td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                <?php endif; ?>
              </div>

              <?php if ($showCreate): ?>
                <div class="modal-backdrop" id="createListModal">
                  <div class="modal-panel">
                    <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:12px;">
                      <div style="font-weight:600;">Create Contact List</div>
                      <button class="btn btn-outline" onclick="document.getElementById('createListModal').style.display='none'">✕</button>
                    </div>
                    <form method="post">
                      <input type="hidden" name="action" value="create_contact_list">
                      <div class="form-group">
                        <label>List name</label>
                        <input type="text" name="list_name" required>
                      </div>
                      <div style="display:flex; gap:8px; justify-content:flex-end;">
                        <a href="?page=contacts" class="btn btn-outline">Cancel</a>
                        <button class="btn btn-primary" type="submit">Create</button>
                      </div>
                    </form>
                  </div>
                </div>
                <script>document.getElementById('createListModal').style.display='flex';</script>
              <?php endif; ?>

            <?php else: ?>
              <?php
                $list = get_contact_list($pdo, $listId);
                if (!$list) {
                  echo "<p>List not found.</p>";
                } else {
                  $contacts = get_contacts_for_list($pdo, $listId);
              ?>
                <div class="page-title">Contacts — <?php echo h($list['name']); ?></div>
                <div class="page-subtitle">Manage recipients in this list (manual add or CSV upload).</div>

                <div class="card">
                  <div class="card-header">
                    <div>
                      <div class="card-title">Add Contacts</div>
                      <div class="card-subtitle">Manual add or upload a CSV file (first column header 'email' is recommended).</div>
                    </div>
                    <div style="display:flex; gap:8px;">
                      <a href="?page=contacts" class="btn btn-outline">← Back to Lists</a>
                    </div>
                  </div>

                  <form method="post" style="margin-bottom:12px;">
                    <input type="hidden" name="action" value="add_contact_manual">
                    <input type="hidden" name="list_id" value="<?php echo (int)$listId; ?>">
                    <div class="form-row">
                      <div class="form-group">
                        <label>Email</label>
                        <input type="email" name="email" required>
                      </div>
                      <div class="form-group">
                        <input type="text" name="first_name" placeholder="First name">
                      </div>
                      <div class="form-group">
                        <label>Last name</label>
                        <input type="text" name="last_name" placeholder="Last name">
                      </div>
                    </div>
                    <div style="display:flex; gap:8px; justify-content:flex-end;">
                      <button type="submit" class="btn btn-primary">Add Contact</button>
                    </div>
                  </form>

                  <form method="post" enctype="multipart/form-data" style="margin-top:12px;">
                    <input type="hidden" name="action" value="upload_contacts_csv">
                    <input type="hidden" name="list_id" value="<?php echo (int)$listId; ?>">
                    <div class="form-row">
                      <div class="form-group">
                        <label>Upload CSV</label>
                        <input type="file" name="csv_file" accept=".csv,text/csv">
                        <small class="hint">CSV should contain an 'email' column or have emails in first column.</small>
                      </div>
                    </div>
                    <div style="display:flex; gap:8px; justify-content:flex-end;">
                      <button class="btn btn-outline" type="submit">Upload</button>
                    </div>
                  </form>
                </div>

                <div class="card">
                  <div class="card-header">
                    <div>
                      <div class="card-title">Contacts (<?php echo count($contacts); ?>)</div>
                    </div>
                    <div>
                      <form method="get" style="display:inline;">
                        <input type="hidden" name="page" value="contacts">
                        <input type="hidden" name="list_id" value="<?php echo (int)$listId; ?>">
                        <input type="text" name="q" placeholder="Search email or name" style="padding:6px 8px;">
                        <button class="btn btn-outline" type="submit">Search</button>
                      </form>
                    </div>
                  </div>

                  <?php if (empty($contacts)): ?>
                    <div class="hint" style="padding:16px;">No contacts in this list yet.</div>
                  <?php else: ?>
                    <table class="table">
                      <thead>
                        <tr>
                          <th>Email</th>
                          <th>Name</th>
                          <th>Added</th>
                        </tr>
                      </thead>
                      <tbody>
                        <?php foreach ($contacts as $ct): ?>
                          <tr>
                            <td><?php echo h($ct['email']); ?></td>
                            <td><?php echo h(trim($ct['first_name'] . ' ' . $ct['last_name'])); ?></td>
                            <td><?php echo h($ct['created_at']); ?></td>
                          </tr>
                        <?php endforeach; ?>
                      </tbody>
                    </table>
                  <?php endif; ?>
                </div>

              <?php } ?>
            <?php endif; ?>

          <?php elseif ($page === 'activity'): ?>
            <?php
              // Activity page showing unified unsubscribes and bounces
              $stmt = $pdo->query("SELECT * FROM events WHERE event_type IN ('unsubscribe','bounce','skipped_unsubscribe','profile_disabled') ORDER BY created_at DESC LIMIT 200");
              $acts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            ?>
            <div class="page-title">Activity</div>
            <div class="page-subtitle">Recent unsubscribes, bounces, and profile disables.</div>

            <div class="card">
              <div class="card-header">
                <div>
                  <div class="card-title">Recent Activity</div>
                  <div class="card-subtitle">Click an event to view details (emails, errors).</div>
                </div>
                <div>
                  <a href="?page=list" class="btn btn-outline">← Single Sends</a>
                </div>
              </div>

              <div style="display:flex; gap:8px; margin-bottom:10px;">
                <form method="post" onsubmit="return confirm('Delete ALL unsubscribes?');">
                  <input type="hidden" name="action" value="delete_unsubscribes">
                  <button class="btn btn-outline" type="submit">Clear Unsubscribes</button>
                </form>
                <form method="post" onsubmit="return confirm('Delete ALL bounce events?');">
                  <input type="hidden" name="action" value="delete_bounces">
                  <button class="btn btn-outline" type="submit">Clear Bounces</button>
                </form>
              </div>

              <?php if (empty($acts)): ?>
                <div class="hint" style="padding:16px;">No recent unsubscribes or bounces.</div>
              <?php else: ?>
                <table class="table">
                  <thead>
                    <tr>
                      <th>Time</th>
                      <th>Type</th>
                      <th>Summary</th>
                      <th>Details</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php foreach ($acts as $a): 
                      $details = json_decode($a['details'], true);
                      $summary = '';
                      if ($a['event_type'] === 'bounce') {
                        $summary = isset($details['rcpt']) ? h($details['rcpt']) : 'Bounce';
                      } elseif ($a['event_type'] === 'unsubscribe' || $a['event_type'] === 'skipped_unsubscribe') {
                        $summary = isset($details['rcpt']) ? h($details['rcpt']) : 'Unsubscribe';
                      } elseif ($a['event_type'] === 'profile_disabled') {
                        $summary = 'Profile ' . (isset($details['profile_id']) ? h($details['profile_id']) : '');
                      }
                    ?>
                      <tr>
                        <td><?php echo h($a['created_at']); ?></td>
                        <td><?php echo h(ucfirst($a['event_type'])); ?></td>
                        <td><?php echo $summary; ?></td>
                        <td><code style="font-size:11px;"><?php echo h($a['details']); ?></code></td>
                      </tr>
                    <?php endforeach; ?>
                  </tbody>
                </table>
              <?php endif; ?>
            </div>

          <?php else: ?>
            <p>Unknown page.</p>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <script>
      (function(){
        var spOpenBtn = document.getElementById('spOpenBtn');
        var spCloseBtn = document.getElementById('spCloseBtn');
        var spSidebar = document.getElementById('spSidebar');
        var pageWrapper = document.getElementById('pageWrapper');

        function openSidebar() {
          spSidebar.classList.add('open');
          pageWrapper.classList.add('shifted');
        }
        function closeSidebar() {
          spSidebar.classList.remove('open');
          pageWrapper.classList.remove('shifted');
        }

        if (spOpenBtn) spOpenBtn.addEventListener('click', openSidebar);
        if (spCloseBtn) spCloseBtn.addEventListener('click', closeSidebar);

        // Toggle profile fields when switching type
        var pfType = document.getElementById('pf_type');
        if (pfType) {
          pfType.addEventListener('change', function() {
            var v = pfType.value;
            var smtpFields = document.getElementById('pf_smtp_fields');
            var apiFields = document.getElementById('pf_api_fields');
            if (v === 'api') {
              smtpFields.style.display = 'none';
              apiFields.style.display = '';
            } else {
              smtpFields.style.display = '';
              apiFields.style.display = 'none';
            }
          });
        }

        // Open sidebar if editing profile
        <?php if ($editProfile): ?>
          openSidebar();
        <?php endif; ?>

        // Toast helper exposed to global scope
        window.showToast = function(msg, type){
          var t = document.getElementById('globalToast');
          if(!t) return;
          t.innerText = msg;
          t.className = 'toast show ' + (type === 'error' ? 'error' : 'success');
          setTimeout(function(){
            t.className = 'toast';
          }, 2000);
        };

        // Show notifications for query params (e.g., test_sent, sent)
        (function(){
          var params = new URLSearchParams(window.location.search);
          if (params.has('test_sent')) {
            showToast('Test message(s) sent', 'success');
          }
          if (params.has('sent')) {
            showToast('Campaign queued for sending', 'success');
          }
          if (params.has('save_error')) {
            showToast('Save error', 'error');
          }
          if (params.has('no_recipients')) {
            showToast('No recipients to send to (all invalid or unsubscribed)', 'error');
          }
          if (params.has('cleared_unsubscribes')) {
            showToast('All unsubscribes cleared', 'success');
          }
          if (params.has('cleared_bounces')) {
            showToast('All bounces cleared', 'success');
          }
        })();

        // Wire up Check Connection buttons (AJAX)
        document.querySelectorAll('.check-conn-btn').forEach(function(btn){
          btn.addEventListener('click', function(){
            var pid = btn.getAttribute('data-pid');
            var statusEl = document.getElementById('profile-conn-status-' + pid);
            if (!statusEl) return;
            statusEl.innerText = 'Checking...';
            var fd = new FormData();
            fd.append('action', 'check_connection_profile');
            fd.append('profile_id', pid);

            fetch(window.location.pathname + window.location.search, {
              method: 'POST',
              body: fd,
              credentials: 'same-origin'
            }).then(function(r){ return r.json(); })
              .then(function(json){
                if (json && json.ok) {
                  statusEl.style.color = 'green';
                  statusEl.innerText = json.msg || 'OK';
                  showToast('Connection OK', 'success');
                } else {
                  statusEl.style.color = 'red';
                  statusEl.innerText = (json && json.msg) ? json.msg : 'Connection failed';
                  showToast('Connection failed', 'error');
                }
              }).catch(function(err){
                statusEl.style.color = 'red';
                statusEl.innerText = 'Connection error';
                showToast('Connection error', 'error');
              });
          });
        });

      })();
    </script>

    </body>
    </html>
